"use client";

import { useState, useEffect, Suspense } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { ArrowLeft, Eye, EyeOff } from "lucide-react";
import { AuthLayoutSplit } from "@/features/auth/components/AuthLayoutSplit";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { AuthFooterLinks } from "@/features/auth/components/AuthFooterLinks";
import { resetPassword } from "@/features/auth/api/resetPassword";
import { setAccessToken, setUserData } from "@/lib/auth";
import { routes } from "@/config/routes";
import Link from "next/link";

function ResetPasswordPageContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const otpId = searchParams.get("otpId");
  const userId = searchParams.get("userId");
  const email = searchParams.get("email") || "";

  const [password, setPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [passwordError, setPasswordError] = useState("");
  const [confirmPasswordError, setConfirmPasswordError] = useState("");
  const [touched, setTouched] = useState<Record<string, boolean>>({});
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState("");

  useEffect(() => {
    if (!otpId || !userId) {
      router.push(routes.forgotPassword);
    }
  }, [otpId, userId, router]);

  const validatePassword = (passwordValue: string): string => {
    if (!passwordValue || passwordValue.trim().length === 0) {
      return "Password is required";
    }
    if (passwordValue.length < 6) {
      return "Password must be at least 6 characters";
    }
    return "";
  };

  const validateConfirmPassword = (
    confirmPasswordValue: string,
    passwordValue: string
  ): string => {
    if (!confirmPasswordValue || confirmPasswordValue.trim().length === 0) {
      return "Please confirm your password";
    }
    if (confirmPasswordValue !== passwordValue) {
      return "Passwords do not match";
    }
    return "";
  };

  const handlePasswordChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setPassword(value);
    if (touched.password) {
      setPasswordError(validatePassword(value));
    }
    if (touched.confirmPassword && confirmPassword) {
      setConfirmPasswordError(validateConfirmPassword(confirmPassword, value));
    }
  };

  const handleConfirmPasswordChange = (
    e: React.ChangeEvent<HTMLInputElement>
  ) => {
    const value = e.target.value;
    setConfirmPassword(value);
    if (touched.confirmPassword) {
      setConfirmPasswordError(validateConfirmPassword(value, password));
    }
  };

  const handleBlur = (field: string) => {
    setTouched((prev) => ({ ...prev, [field]: true }));
    if (field === "password") {
      setPasswordError(validatePassword(password));
      if (confirmPassword) {
        setConfirmPasswordError(validateConfirmPassword(confirmPassword, password));
      }
    } else if (field === "confirmPassword") {
      setConfirmPasswordError(validateConfirmPassword(confirmPassword, password));
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    // Mark all fields as touched
    setTouched({ password: true, confirmPassword: true });

    // Validate fields
    const passwordErr = validatePassword(password);
    const confirmPasswordErr = validateConfirmPassword(confirmPassword, password);
    setPasswordError(passwordErr);
    setConfirmPasswordError(confirmPasswordErr);

    if (passwordErr || confirmPasswordErr || !password || !confirmPassword) {
      return;
    }

    if (!otpId || !userId) {
      setError("Invalid reset data. Please try again.");
      return;
    }

    setIsLoading(true);
    setError("");

    try {
      const response = await resetPassword(password, userId, otpId);

      if (response.status === "1" || response.status === "2") {
        // Password reset successful - store auth data and redirect
        const userData = response.data;
        setAccessToken(userData.accessToken);
        setUserData({
          accessToken: userData.accessToken,
          userId: userData.userId,
          email: userData.email,
          firstName: userData.firstName,
          lastName: userData.lastName,
          phoneNum: "",
          image: "",
          stripeCustomerId: null,
        });

        // Redirect to home page
        router.push(routes.home);
      } else {
        setError(
          response.message || "Failed to reset password. Please try again."
        );
      }
    } catch (err: any) {
      console.error("Reset password error:", err);
      setError(err.message || "Failed to reset password. Please try again.");
    } finally {
      setIsLoading(false);
    }
  };

  if (!otpId || !userId) {
    return null;
  }

  return (
    <AuthLayoutSplit>
      <div className="max-w-md mx-auto w-full flex flex-col min-h-full">
        <div className="mb-8">
          <button
            onClick={() => router.back()}
            className="mb-6 p-2 -ml-2 hover:bg-gray-50 rounded-lg transition-colors"
            aria-label="Go back"
          >
            <ArrowLeft className="w-5 h-5 text-gray-900" />
          </button>
          <h1 className="heading-2 mb-2">Reset your password</h1>
          <p className="body-text-muted">
            Enter your new password below. Make sure it's at least 6 characters
            long.
          </p>
        </div>

        <form onSubmit={handleSubmit} className="flex flex-col flex-1">
          <div className="space-y-4 mb-6">
            {/* New Password */}
            <div>
              <label
                className={`label block mb-2 ${
                  touched.password && passwordError ? "text-error-500" : ""
                }`}
              >
                New password *
              </label>
              <div className="relative">
                <Input
                  id="password"
                  type={showPassword ? "text" : "password"}
                  placeholder="Enter new password"
                  value={password}
                  onChange={handlePasswordChange}
                  onBlur={() => handleBlur("password")}
                  error={passwordError}
                  touched={touched.password}
                  required
                  className="pr-10"
                  disabled={isLoading}
                />
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-500 hover:text-gray-700"
                  aria-label={showPassword ? "Hide password" : "Show password"}
                >
                  {showPassword ? (
                    <EyeOff className="w-5 h-5" />
                  ) : (
                    <Eye className="w-5 h-5" />
                  )}
                </button>
              </div>
              {touched.password && passwordError && (
                <p className="mt-1 text-sm text-error-500">{passwordError}</p>
              )}
            </div>

            {/* Confirm Password */}
            <div>
              <label
                className={`label block mb-2 ${
                  touched.confirmPassword && confirmPasswordError
                    ? "text-error-500"
                    : ""
                }`}
              >
                Confirm new password *
              </label>
              <div className="relative">
                <Input
                  id="confirmPassword"
                  type={showConfirmPassword ? "text" : "password"}
                  placeholder="Confirm new password"
                  value={confirmPassword}
                  onChange={handleConfirmPasswordChange}
                  onBlur={() => handleBlur("confirmPassword")}
                  error={confirmPasswordError}
                  touched={touched.confirmPassword}
                  required
                  className="pr-10"
                  disabled={isLoading}
                />
                <button
                  type="button"
                  onClick={() =>
                    setShowConfirmPassword(!showConfirmPassword)
                  }
                  className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-500 hover:text-gray-700"
                  aria-label={
                    showConfirmPassword ? "Hide password" : "Show password"
                  }
                >
                  {showConfirmPassword ? (
                    <EyeOff className="w-5 h-5" />
                  ) : (
                    <Eye className="w-5 h-5" />
                  )}
                </button>
              </div>
              {touched.confirmPassword && confirmPasswordError && (
                <p className="mt-1 text-sm text-error-500">
                  {confirmPasswordError}
                </p>
              )}
            </div>
          </div>

          {error && (
            <div className="mb-4 p-3 rounded-xl bg-error-50 border border-error-200">
              <p className="body-text-sm text-error-500">{error}</p>
            </div>
          )}

          <Button
            type="submit"
            disabled={
              !password ||
              !confirmPassword ||
              !!passwordError ||
              !!confirmPasswordError ||
              isLoading
            }
            loading={isLoading}
            className="w-full mb-auto"
          >
            Reset password
          </Button>

          <div className="text-center mb-auto">
            <Link
              href={routes.login}
              className="body-text-sm text-primary-500 hover:underline font-medium"
            >
              Back to login
            </Link>
          </div>

          <AuthFooterLinks />
        </form>
      </div>
    </AuthLayoutSplit>
  );
}

export default function ResetPasswordPage() {
  return (
    <Suspense fallback={
      <AuthLayoutSplit>
        <div className="max-w-md mx-auto w-full flex flex-col min-h-full items-center justify-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900"></div>
        </div>
      </AuthLayoutSplit>
    }>
      <ResetPasswordPageContent />
    </Suspense>
  );
}




