"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";
import { AuthLayoutSplit } from "@/features/auth/components/AuthLayoutSplit";
import { AuthHeader } from "@/features/auth/components/AuthHeader";
import { SocialLoginButton } from "@/features/auth/components/SocialLoginButton";
import { OrDivider } from "@/features/auth/components/OrDivider";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { AuthFooterLinks } from "@/features/auth/components/AuthFooterLinks";
import { routes } from "@/config/routes";
import { login } from "@/features/auth/api/login";
import { setAccessToken, setUserData } from "@/lib/auth";
import { handleSocialAuth } from "@/features/auth/api/socialAuth";
import Link from "next/link";
import { Eye, EyeOff } from "lucide-react";

export default function LoginPage() {
  const router = useRouter();
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const [emailError, setEmailError] = useState("");
  const [passwordError, setPasswordError] = useState("");
  const [touched, setTouched] = useState<Record<string, boolean>>({});
  const [isLoading, setIsLoading] = useState(false);

  const validateEmail = (emailValue: string): string => {
    if (!emailValue || emailValue.trim().length === 0) {
      return "Email address is required";
    }
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(emailValue.trim())) {
      return "Please enter a valid email address";
    }
    return "";
  };

  const validatePassword = (passwordValue: string): string => {
    if (!passwordValue || passwordValue.trim().length === 0) {
      return "Password is required";
    }
    return "";
  };

  const handleEmailChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setEmail(value);
    // Clear error when user starts typing
    if (emailError) {
      setEmailError("");
    }
    if (touched.email) {
      setEmailError(validateEmail(value));
    }
  };

  const handlePasswordChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setPassword(value);
    // Clear error when user starts typing
    if (passwordError) {
      setPasswordError("");
    }
    if (touched.password) {
      setPasswordError(validatePassword(value));
    }
  };

  const handleBlur = (field: string) => {
    setTouched((prev) => ({ ...prev, [field]: true }));
    if (field === "email") {
      setEmailError(validateEmail(email));
    } else if (field === "password") {
      setPasswordError(validatePassword(password));
    }
  };

  const handleLogin = async (e?: React.FormEvent) => {
    if (e) {
      e.preventDefault();
    }

    // Mark all fields as touched
    setTouched({ email: true, password: true });

    // Validate fields
    const emailErr = validateEmail(email);
    const passwordErr = validatePassword(password);
    setEmailError(emailErr);
    setPasswordError(passwordErr);

    if (emailErr || passwordErr || !email || !password) {
      return;
    }

    setIsLoading(true);
    try {
      const response = await login(email.trim(), password);

      if (response.status === "1") {
        // User is verified - login successful
        const userData = response.data as any;
        if (userData && userData.accessToken) {
          setAccessToken(userData.accessToken);
          setUserData(userData);
          router.push(routes.home);
        } else {
          setEmailError(
            "Login successful but user data is missing. Please try again."
          );
        }
      } else if (response.status === "2") {
        // User is not verified - redirect to OTP screen
        const userId = response.data?.userId;
        if (userId) {
          router.push(
            `/signup/verify-otp?userId=${userId}&email=${encodeURIComponent(
              email
            )}&login=true`
          );
        } else {
          setEmailError(
            "Account verification required, but user ID is missing. Please contact support."
          );
        }
      } else if (response.status === "0") {
        // Invalid email or user not found
        setEmailError(
          response.message || "Invalid email or password. Please try again."
        );
      } else {
        setEmailError(
          response.message || "Failed to log in. Please try again."
        );
      }
    } catch (err: any) {
      console.error("Login error:", err);
      setEmailError(err.message || "Failed to log in. Please try again.");
    } finally {
      setIsLoading(false);
    }
  };

  const handleSocialLogin = async (provider: "google" | "apple") => {
    setIsLoading(true);
    try {
      const result = await handleSocialAuth(provider);

      if (result.success) {
        if (result.requiresSignup && result.userData) {
          // Data is incomplete - store in sessionStorage and redirect to signup form
          const { setSocialAuthData } = await import("@/lib/socialAuthStorage");
          setSocialAuthData(result.userData);
          router.push(`${routes.signupCreate}?socialAuth=true`);
        } else if (result.requiresOtp) {
          // If OTP is required, redirect to OTP page
          router.push(routes.signupVerifyOtp);
        } else {
          // User is logged in, redirect to home
          router.push(routes.home);
        }
      } else {
        setEmailError(
          result.error || "Authentication failed. Please try again."
        );
      }
    } catch (error: any) {
      console.error("Social login error:", error);
      setEmailError(
        error.message || "Authentication failed. Please try again."
      );
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <AuthLayoutSplit>
      <div className="max-w-md mx-auto w-full flex flex-col min-h-full">
        <AuthHeader
          title="Fresha for customers"
          subtitle="Create an account or log in to book and manage your appointments."
        />

        {/* Social Login Buttons */}
        <div className="space-y-3 mb-6">
          <SocialLoginButton
            provider="google"
            onClick={() => handleSocialLogin("google")}
            loading={isLoading}
          >
            Continue with Google
          </SocialLoginButton>
          <SocialLoginButton
            provider="apple"
            onClick={() => handleSocialLogin("apple")}
            loading={isLoading}
          >
            Continue with Apple
          </SocialLoginButton>
        </div>

        <OrDivider />

        {/* Login Form */}
        <form 
          onSubmit={handleLogin} 
          className="flex flex-col flex-1"
          data-testid="login-form"
          noValidate
        >
          <div className="space-y-4 mb-6">
            {/* Email Input */}
            <Input
              id="email"
              name="email"
              type="email"
              label="Email address"
              placeholder="Email address"
              value={email}
              onChange={handleEmailChange}
              onInput={handleEmailChange}
              onBlur={() => handleBlur("email")}
              error={emailError}
              touched={touched.email}
              required
              autoComplete="email"
              data-testid="login-email-input"
            />

            {/* Password Input */}
            <div>
              <label
                className={`label block mb-2 ${
                  touched.password && passwordError ? "text-error-500" : ""
                }`}
              >
                Password *
              </label>
              <div className="relative">
                <Input
                  id="password"
                  name="password"
                  type={showPassword ? "text" : "password"}
                  placeholder="Password"
                  value={password}
                  onChange={handlePasswordChange}
                  onInput={handlePasswordChange}
                  onBlur={() => handleBlur("password")}
                  error={passwordError}
                  touched={touched.password}
                  required
                  className="pr-10"
                  autoComplete="current-password"
                  data-testid="login-password-input"
                />
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-500 hover:text-gray-700"
                  aria-label={showPassword ? "Hide password" : "Show password"}
                >
                  {showPassword ? (
                    <EyeOff className="w-5 h-5" />
                  ) : (
                    <Eye className="w-5 h-5" />
                  )}
                </button>
              </div>
              {touched.password && passwordError && (
                <p className="mt-1 text-sm text-error-500">{passwordError}</p>
              )}
            </div>

            {/* Forgot Password Link */}
            <div className="text-right">
              <Link
                href="/forgot-password"
                className="body-text-sm text-primary-500 hover:underline font-medium"
              >
                Forgot password?
              </Link>
            </div>
          </div>

          {/* Login Button */}
          <Button
            type="submit"
            disabled={!email || !password || !!emailError || !!passwordError}
            loading={isLoading}
            className="w-full mb-6"
            data-testid="login-submit-button"
          >
            Log in
          </Button>
        </form>

        {/* Business Account Link */}
        <p className="body-text text-center mb-auto">
          Have a business account?{" "}
          <Link
            href="/login/professional"
            className="text-primary-500 hover:underline font-medium"
          >
            Sign in as a professional
          </Link>
        </p>

        {/* Footer Links */}
        <AuthFooterLinks />
      </div>
    </AuthLayoutSplit>
  );
}
