"use client";

import { useState, useEffect, Suspense } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { ArrowLeft } from "lucide-react";
import { AuthLayoutSplit } from "@/features/auth/components/AuthLayoutSplit";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { AuthFooterLinks } from "@/features/auth/components/AuthFooterLinks";
import { verifyOtpForgotPassword } from "@/features/auth/api/verifyOtpForgotPassword";
import { resendOtp } from "@/features/auth/api/resendOtp";
import { routes } from "@/config/routes";
import Link from "next/link";

function ForgotPasswordVerifyOtpContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const otpIdParam = searchParams.get("otpId");
  const userIdParam = searchParams.get("userId");
  const email = searchParams.get("email") || "";

  const [otp, setOtp] = useState("");
  const [error, setError] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [isResending, setIsResending] = useState(false);
  const [otpId, setOtpId] = useState<string | null>(otpIdParam || null);
  const [userId, setUserId] = useState<string | null>(userIdParam || null);
  const [otpSent, setOtpSent] = useState(false);

  useEffect(() => {
    if (!userIdParam || !otpIdParam) {
      router.push(routes.forgotPassword);
    }
  }, [userIdParam, otpIdParam, router]);

  const handleOtpChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value.replace(/\D/g, "").slice(0, 6);
    setOtp(value);
    setError(""); // Clear error when user starts typing
  };

  const handleResendOtp = async () => {
    if (!userId || !email) {
      setError("Unable to resend OTP. Please try again.");
      return;
    }

    setIsResending(true);
    setError("");

    try {
      const response = await resendOtp(email, userId);

      if (response.status === "1" || response.status === "2") {
        // Successfully sent OTP
        setOtpId(response.data.otpId);
        setUserId(response.data.userId);
        setOtpSent(true);
        setError(""); // Clear any previous errors

        // Show success message
        setTimeout(() => {
          setOtpSent(false);
        }, 3000);
      } else {
        setError(
          response.message || "Failed to send verification code. Please try again."
        );
      }
    } catch (err: any) {
      console.error("Resend OTP error:", err);
      setError(
        err.message || "Failed to send verification code. Please try again."
      );
    } finally {
      setIsResending(false);
    }
  };

  const handleVerify = async () => {
    if (!otp || otp.length < 4) {
      setError("Please enter a valid verification code");
      return;
    }

    if (!otpId) {
      setError(
        "Verification code not available. Please click 'Resend code' to receive a new one."
      );
      return;
    }

    setIsLoading(true);
    setError("");

    try {
      const response = await verifyOtpForgotPassword(otpId, otp);

      if (response.status === "1" || response.status === "2") {
        // OTP verified successfully - redirect to reset password page
        router.push(
          `/reset-password?otpId=${response.data.otpId}&userId=${response.data.userId}&email=${encodeURIComponent(email)}`
        );
      } else {
        setError(
          response.message || "Invalid verification code. Please try again."
        );
        setOtp(""); // Clear OTP on error
      }
    } catch (err: any) {
      console.error("OTP verification error:", err);
      // Clear OTP input on error so user can re-enter
      setOtp("");

      // Show specific error message
      if (
        err.message?.toLowerCase().includes("invalid") ||
        err.message?.toLowerCase().includes("incorrect") ||
        err.message?.toLowerCase().includes("wrong")
      ) {
        setError(
          "The verification code you entered is incorrect. Please check your email and try again, or click 'Resend code' to receive a new one."
        );
      } else {
        setError(
          err.message ||
            "Verification failed. Please try again or click 'Resend code' to receive a new verification code."
        );
      }
    } finally {
      setIsLoading(false);
    }
  };

  if (!userId || !otpId) {
    return null;
  }

  return (
    <AuthLayoutSplit>
      <div className="max-w-md mx-auto w-full flex flex-col min-h-full">
        <div className="mb-8">
          <button
            onClick={() => router.back()}
            className="mb-6 p-2 -ml-2 hover:bg-gray-50 rounded-lg transition-colors"
            aria-label="Go back"
          >
            <ArrowLeft className="w-5 h-5 text-gray-900" />
          </button>
          <h1 className="heading-2 mb-2">Verify your email</h1>
          <p className="body-text-muted">
            {otpSent
              ? "A new verification code has been sent to "
              : "We've sent a verification code to "}
            <span className="font-semibold text-primary-500">{email}</span>
            {otpSent && (
              <span className="block mt-1 text-sm text-green-600">
                Code sent successfully!
              </span>
            )}
          </p>
        </div>

        <form
          onSubmit={(e) => {
            e.preventDefault();
            handleVerify();
          }}
          className="flex flex-col flex-1"
        >
          <div className="mb-6">
            <Input
              id="otp"
              label="Enter verification code"
              type="text"
              inputMode="numeric"
              value={otp}
              onChange={handleOtpChange}
              placeholder="000000"
              maxLength={6}
              error={error}
              touched={!!error}
              required
              className="text-center text-2xl tracking-widest font-semibold"
              disabled={isLoading}
            />
            <p className="body-text-sm text-muted mt-2 text-center">
              Enter the 4-6 digit code sent to your email
            </p>
          </div>

          {error && (
            <div className="mb-4 p-3 rounded-xl bg-error-50 border border-error-200">
              <p className="body-text-sm text-error-500">{error}</p>
            </div>
          )}

          <Button
            type="submit"
            disabled={!otp || otp.length < 4 || isLoading}
            loading={isLoading}
            className="w-full mb-auto"
          >
            Verify
          </Button>

          <div className="text-center mb-auto">
            <p className="body-text-sm text-muted mb-2">
              Didn't receive the code?
            </p>
            <button
              type="button"
              onClick={handleResendOtp}
              disabled={isResending || isLoading}
              className="body-text-sm text-primary-500 hover:underline font-medium disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isResending ? "Sending..." : "Resend code"}
            </button>
          </div>

          <div className="text-center mb-auto">
            <Link
              href={routes.login}
              className="body-text-sm text-primary-500 hover:underline font-medium"
            >
              Back to login
            </Link>
          </div>

          <AuthFooterLinks />
        </form>
      </div>
    </AuthLayoutSplit>
  );
}

export default function ForgotPasswordVerifyOtpPage() {
  return (
    <Suspense fallback={
      <AuthLayoutSplit>
        <div className="max-w-md mx-auto w-full flex flex-col min-h-full items-center justify-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900"></div>
        </div>
      </AuthLayoutSplit>
    }>
      <ForgotPasswordVerifyOtpContent />
    </Suspense>
  );
}




