"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";
import { ArrowLeft } from "lucide-react";
import { AuthLayoutSplit } from "@/features/auth/components/AuthLayoutSplit";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { AuthFooterLinks } from "@/features/auth/components/AuthFooterLinks";
import { forgotPassword } from "@/features/auth/api/forgotPassword";
import { routes } from "@/config/routes";
import Link from "next/link";

export default function ForgotPasswordPage() {
  const router = useRouter();
  const [email, setEmail] = useState("");
  const [emailError, setEmailError] = useState("");
  const [touched, setTouched] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [success, setSuccess] = useState(false);

  const validateEmail = (emailValue: string): string => {
    if (!emailValue || emailValue.trim().length === 0) {
      return "Email address is required";
    }
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(emailValue.trim())) {
      return "Please enter a valid email address";
    }
    return "";
  };

  const handleEmailChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setEmail(value);
    if (touched) {
      setEmailError(validateEmail(value));
    }
  };

  const handleEmailBlur = () => {
    setTouched(true);
    setEmailError(validateEmail(email));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    setTouched(true);
    const error = validateEmail(email);
    setEmailError(error);

    if (error || !email) {
      return;
    }

    setIsLoading(true);
    setEmailError("");

    try {
      const response = await forgotPassword(email.trim());

      if (response.status === "1" || response.status === "2") {
        // Successfully sent OTP
        setSuccess(true);
        // Redirect to OTP verification page
        router.push(
          `/forgot-password/verify-otp?otpId=${response.data.otpId}&userId=${response.data.userId}&email=${encodeURIComponent(email)}`
        );
      } else {
        setEmailError(
          response.message || "Failed to send verification code. Please try again."
        );
      }
    } catch (err: any) {
      console.error("Forgot password error:", err);
      setEmailError(
        err.message || "Failed to send verification code. Please try again."
      );
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <AuthLayoutSplit>
      <div className="max-w-md mx-auto w-full flex flex-col min-h-full">
        <div className="mb-8">
          <button
            onClick={() => router.back()}
            className="mb-6 p-2 -ml-2 hover:bg-gray-50 rounded-lg transition-colors"
            aria-label="Go back"
          >
            <ArrowLeft className="w-5 h-5 text-gray-900" />
          </button>
          <h1 className="heading-2 mb-2">Reset your password</h1>
          <p className="body-text-muted">
            Enter your email address and we'll send you a verification code to
            reset your password.
          </p>
        </div>

        <form onSubmit={handleSubmit} className="flex flex-col flex-1">
          <div className="mb-6">
            <Input
              id="email"
              type="email"
              label="Email address"
              placeholder="Email address"
              value={email}
              onChange={handleEmailChange}
              onBlur={handleEmailBlur}
              error={emailError}
              touched={touched}
              required
              disabled={isLoading || success}
            />
          </div>

          {emailError && (
            <div className="mb-4 p-3 rounded-xl bg-error-50 border border-error-200">
              <p className="body-text-sm text-error-500">{emailError}</p>
            </div>
          )}

          <Button
            type="submit"
            disabled={!email || !!emailError || isLoading || success}
            loading={isLoading}
            className="w-full mb-auto"
          >
            {success ? "Code sent!" : "Send verification code"}
          </Button>

          <div className="text-center mb-auto">
            <p className="body-text text-muted">
              Remember your password?{" "}
              <Link
                href={routes.login}
                className="text-primary-500 hover:underline font-medium"
              >
                Back to login
              </Link>
            </p>
          </div>

          <AuthFooterLinks />
        </form>
      </div>
    </AuthLayoutSplit>
  );
}




