# Typography System Guide

This guide explains how to use the semantic typography classes in the Fresha project. **Always use these classes instead of defining typography properties individually.**

## Philosophy

Instead of writing:
```tsx
<h1 className="text-3xl font-bold text-gray-900 leading-tight" style={{ color: "var(--text-primary)" }}>
  Title
</h1>
```

Use:
```tsx
<h1 className="heading-1">
  Title
</h1>
```

## Available Typography Classes

### Headings

| Class | Font Size | Weight | Line Height | Use Case |
|-------|-----------|--------|-------------|----------|
| `.heading-1` | 3rem (48px) | 700 | 1.25 | Main page titles |
| `.heading-2` | 2.25rem (36px) | 700 | 1.25 | Section titles |
| `.heading-3` | 1.875rem (30px) | 700 | 1.25 | Subsection titles |
| `.heading-4` | 1.5rem (24px) | 600 | 1.25 | Card titles, smaller sections |
| `.heading-5` | 1.25rem (20px) | 600 | 1.5 | Small headings |
| `.heading-6` | 1.125rem (18px) | 600 | 1.5 | Smallest headings |
| `.heading-hero` | Responsive | 700 | 1.25 | Hero section titles (scales 48px → 60px → 72px) |

**Example:**
```tsx
<h1 className="heading-1">Main Title</h1>
<h2 className="heading-2">Section Title</h2>
<h3 className="heading-3">Subsection</h3>
```

### Body Text

| Class | Font Size | Weight | Line Height | Color |
|-------|-----------|--------|-------------|-------|
| `.body-text` | 1rem (16px) | 400 | 1.5 | Primary text |
| `.body-text-lg` | 1.125rem (18px) | 400 | 1.5 | Larger body text |
| `.body-text-sm` | 0.875rem (14px) | 400 | 1.5 | Smaller body text |
| `.body-text-muted` | 1rem (16px) | 400 | 1.5 | Muted primary text |
| `.body-text-muted-lg` | 1.125rem (18px) | 400 | 1.5 | Muted larger text |
| `.body-text-muted-sm` | 0.875rem (14px) | 400 | 1.5 | Muted smaller text |

**Example:**
```tsx
<p className="body-text">Regular paragraph text</p>
<p className="body-text-muted">Secondary information</p>
<p className="body-text-lg">Larger paragraph</p>
```

### Captions & Labels

| Class | Font Size | Weight | Line Height | Use Case |
|-------|-----------|--------|-------------|----------|
| `.caption` | 0.75rem (12px) | 400 | 1.5 | Small captions, helper text |
| `.caption-bold` | 0.75rem (12px) | 600 | 1.5 | Bold captions |
| `.label` | 0.875rem (14px) | 500 | 1.5 | Form labels, tags |
| `.label-sm` | 0.75rem (12px) | 500 | 1.5 | Small labels |

**Example:**
```tsx
<span className="caption">Helper text</span>
<label className="label">Form Label</label>
<span className="caption-bold">Bold Caption</span>
```

### Card Text

| Class | Font Size | Weight | Line Height | Use Case |
|-------|-----------|--------|-------------|----------|
| `.card-title` | 1rem (16px) | 600 | 1.25 | Venue/product names |
| `.card-text` | 0.875rem (14px) | 400 | 1.5 | Card descriptions, addresses |
| `.card-category` | 0.75rem (12px) | 400 | 1.5 | Categories, tags |

**Example:**
```tsx
<h3 className="card-title">Venue Name</h3>
<p className="card-text">Address or description</p>
<span className="card-category">Category</span>
```

### Button Text

| Class | Font Size | Weight | Line Height | Use Case |
|-------|-----------|--------|-------------|----------|
| `.button-text` | 1rem (16px) | 500 | 1.5 | Standard buttons |
| `.button-text-sm` | 0.875rem (14px) | 500 | 1.5 | Small buttons |
| `.button-text-lg` | 1.125rem (18px) | 600 | 1.5 | Large buttons, CTAs |

**Example:**
```tsx
<button className="button-text">Click Me</button>
<button className="button-text-lg">Large CTA</button>
```

### Navigation Text

| Class | Font Size | Weight | Line Height | Use Case |
|-------|-----------|--------|-------------|----------|
| `.nav-link` | 1rem (16px) | 500 | 1.5 | Navigation links |
| `.nav-link-sm` | 0.875rem (14px) | 500 | 1.5 | Small nav links |

**Example:**
```tsx
<Link className="nav-link">Home</Link>
<Link className="nav-link-sm">Small Link</Link>
```

## Utility Modifiers

You can combine typography classes with utility modifiers:

### Color Modifiers
```tsx
<p className="body-text text-muted">Muted text</p>
```

### Weight Modifiers (if you need to override)
```tsx
<p className="body-text font-semibold">Bold body text</p>
```

### Alignment
```tsx
<h2 className="heading-2 text-center">Centered Heading</h2>
```

### Text Transform
```tsx
<span className="label uppercase">UPPERCASE LABEL</span>
```

## Migration Guide

### Before (Don't Do This)
```tsx
<h1 className="text-3xl md:text-4xl font-bold text-gray-900 mb-8" style={{ color: "var(--text-primary)" }}>
  Title
</h1>
<p className="text-lg text-gray-600" style={{ color: "var(--text-muted)" }}>
  Description
</p>
```

### After (Do This)
```tsx
<h1 className="heading-2 mb-8">
  Title
</h1>
<p className="body-text-muted-lg">
  Description
</p>
```

## Responsive Typography

For responsive typography, use Tailwind's responsive prefixes with the typography classes:

```tsx
<h1 className="heading-3 md:heading-2 lg:heading-1">
  Responsive Heading
</h1>
```

Or use the built-in `.heading-hero` class which is already responsive.

## Best Practices

1. **Always use semantic classes** - Don't define font-size, font-weight, color individually
2. **Combine with spacing utilities** - Typography classes handle text styling, spacing classes handle layout
3. **Use appropriate semantic class** - Choose the class that matches the content's purpose
4. **Override only when necessary** - Use utility modifiers sparingly
5. **Maintain consistency** - Use the same class for similar content across the app

## Examples

### Section Header
```tsx
<section>
  <h2 className="heading-2 mb-4">Section Title</h2>
  <p className="body-text-muted mb-8">Section description</p>
</section>
```

### Card Component
```tsx
<div className="card">
  <h3 className="card-title mb-2">Card Title</h3>
  <p className="card-text mb-1">Card description</p>
  <span className="card-category">Category</span>
</div>
```

### Button
```tsx
<button className="button-text px-4 py-2 bg-primary-500 text-white rounded-full">
  Click Me
</button>
```

### Navigation
```tsx
<nav>
  <Link className="nav-link">Home</Link>
  <Link className="nav-link">About</Link>
</nav>
```




