# Responsive Design Breakpoints

This document outlines the responsive breakpoints used throughout the Fresha project.

## Breakpoint System

The project uses **Tailwind CSS default breakpoints**, which are defined in `src/config/theme.ts`:

| Breakpoint | Min Width | Device Type | Usage |
|------------|-----------|-------------|-------|
| **Default** | 0px | Mobile (portrait) | Base styles, no prefix |
| **sm** | 640px | Mobile (landscape) / Small tablets | `sm:` prefix |
| **md** | 768px | Tablets | `md:` prefix |
| **lg** | 1024px | Small laptops / Desktops | `lg:` prefix |
| **xl** | 1280px | Large desktops | `xl:` prefix |
| **2xl** | 1536px | Extra large desktops | `2xl:` prefix |

## Breakpoint Strategy

### Mobile-First Approach
- **Base styles** (no prefix) target mobile devices (0px+)
- **Progressive enhancement** using `sm:`, `md:`, `lg:`, `xl:`, `2xl:` prefixes
- Styles are applied at the breakpoint and above

### Example Usage
```tsx
// Mobile-first: starts at 0px, changes at 768px
<div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3">
  {/* 1 column on mobile, 2 on tablet, 3 on desktop */}
</div>

// Typography scaling
<h1 className="text-3xl sm:text-4xl md:text-5xl lg:text-6xl">
  {/* Scales from 3xl → 4xl → 5xl → 6xl */}
</h1>
```

## Component-Specific Responsive Patterns

### 1. Header (`src/components/navigation/Header.tsx`)
- **Mobile**: Hamburger menu button visible, navigation hidden
- **Tablet+ (md)**: Full navigation visible, hamburger hidden
- **Breakpoints**: `md:hidden`, `hidden md:flex`

### 2. Hero Section (`src/features/home/components/HeroSection.tsx`)
- **Mobile**: 
  - Smaller heading (text-3xl)
  - Stacked search inputs (flex-col)
  - Reduced padding
- **Tablet+ (md)**: 
  - Horizontal search bar (flex-row)
  - Dividers visible
- **Desktop (lg+)**: 
  - Larger heading (text-6xl, text-7xl)
  - Full spacing

### 3. Venue Cards (`src/features/home/components/VenueCard.tsx`)
- **Card Width**: Fixed at 296px (theme token)
- **Carousel**: Horizontal scroll on all devices
- **Grid**: Not used (carousel pattern)

### 4. Reviews Section (`src/features/home/components/ReviewsSection.tsx`)
- **Mobile**: 
  - Card width: 280px
  - Arrows hidden
  - Reduced gap (gap-4)
- **Tablet (sm)**: 
  - Card width: 320px
- **Desktop (md+)**: 
  - Card width: 380px
  - Arrows visible
  - Full gap (gap-6)

### 5. Stats Section (`src/features/home/components/TopStatsSection.tsx`)
- **Mobile**: 
  - Heading: text-3xl
  - Big number: text-5xl
  - Stats: 1 column, text-4xl
- **Tablet (sm)**: 
  - Stats: 3 columns
  - Larger numbers
- **Desktop (md+)**: 
  - Heading: text-5xl → text-6xl
  - Big number: text-7xl → text-9xl → text-[10rem]
  - Stats: text-5xl → text-6xl

### 6. Business Promo Section (`src/features/home/components/BusinessPromoSection.tsx`)
- **Mobile**: 
  - Single column layout
  - Smaller heading (text-3xl)
  - Phone mockup: w-32, h-[250px]
  - Reduced padding
- **Tablet (md)**: 
  - Two column layout
  - Larger heading (text-5xl)
  - Phone mockup: w-48, h-[380px]
- **Desktop (lg+)**: 
  - Heading: text-6xl
  - Full spacing

### 7. City Directory (`src/features/home/components/CityDirectorySection.tsx`)
- **Mobile**: 
  - 1 column grid
  - Right arrow hidden
  - Scrollable country pills
- **Tablet (sm)**: 
  - 2 columns
- **Tablet (md)**: 
  - 3 columns, arrow visible
- **Desktop (lg+)**: 
  - 4-5 columns

### 8. App Download Section (`src/features/home/components/AppDownloadSection.tsx`)
- **Mobile**: 
  - Single column
  - Phone mockups hidden
  - Smaller heading
- **Tablet+ (md)**: 
  - Two column layout
  - Phone mockups visible
  - Full sizing

### 9. Footer (`src/components/navigation/Footer.tsx`)
- **Mobile**: 
  - 2 columns
  - Brand column spans 2
- **Tablet (sm)**: 
  - 3 columns
- **Desktop (md+)**: 
  - 5 columns
  - Brand column spans 1

## Common Responsive Patterns

### Typography Scaling
```tsx
// Headings
text-3xl sm:text-4xl md:text-5xl lg:text-6xl

// Body text
text-base sm:text-lg md:text-xl

// Small text
text-sm md:text-base
```

### Spacing
```tsx
// Padding
py-12 md:py-16 lg:py-20

// Gaps
gap-4 md:gap-6 lg:gap-8

// Margins
mb-4 md:mb-6 lg:mb-8
```

### Grid Layouts
```tsx
// Responsive columns
grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5

// Two-column layouts
grid-cols-1 md:grid-cols-2
```

### Visibility
```tsx
// Hide on mobile, show on desktop
hidden md:block

// Show on mobile, hide on desktop
block md:hidden
```

### Container Widths
```tsx
// Max widths
max-w-md sm:max-w-lg md:max-w-xl lg:max-w-2xl

// Full width with padding
w-full px-4 sm:px-6 md:px-8
```

## Testing Recommendations

1. **Mobile (320px - 639px)**: Test on iPhone SE, small Android devices
2. **Tablet Portrait (640px - 767px)**: Test on iPad Mini, small tablets
3. **Tablet Landscape (768px - 1023px)**: Test on iPad, medium tablets
4. **Desktop (1024px - 1279px)**: Test on laptops, small desktops
5. **Large Desktop (1280px+)**: Test on large monitors

## Theme Configuration

Breakpoints are defined in `src/config/theme.ts`:

```typescript
breakpoints: {
  sm: "640px",
  md: "768px",
  lg: "1024px",
  xl: "1280px",
  "2xl": "1536px",
}
```

These values match Tailwind CSS defaults and are used for reference only. Tailwind automatically handles the responsive classes.

## Best Practices

1. **Always start mobile-first**: Write base styles for mobile, then enhance for larger screens
2. **Use semantic breakpoints**: Choose breakpoints based on content needs, not device sizes
3. **Test on real devices**: Emulators are helpful, but real device testing is essential
4. **Consider touch targets**: Ensure interactive elements are at least 44x44px on mobile
5. **Optimize images**: Use responsive images with `next/image` and appropriate sizes
6. **Hide decorative elements on mobile**: Arrows, overlays, and complex visuals can be hidden on small screens
7. **Maintain readability**: Font sizes should never be too small on any device




