# Project Structure Quick Reference

## 🗂️ Directory Map

```
src/
├── app/              → Next.js routes (pages & layouts)
├── components/       → Reusable UI (design system)
├── features/         → Domain features (business logic)
├── lib/              → Utilities (pure functions)
├── hooks/            → Shared React hooks
├── config/           → Configuration & constants
├── types/            → TypeScript types
├── store/            → Zustand state
└── styles/           → Global CSS
```

## 📍 Where to Put Code

| Type of Code | Location | Example |
|-------------|----------|---------|
| **Reusable Button** | `src/components/ui/button.tsx` | Generic button component |
| **Header Component** | `src/components/navigation/Header.tsx` | Site header |
| **Home Hero Section** | `src/features/home/components/HeroSection.tsx` | Home page hero |
| **Venue Detail Page** | `src/features/venues/components/VenueHeader.tsx` | Venue-specific component |
| **Booking Step** | `src/features/booking/components/TimeStep.tsx` | Booking flow step |
| **Format Function** | `src/lib/formatCurrency.ts` | Currency formatter |
| **Shared Hook** | `src/hooks/useToggle.ts` | Toggle hook |
| **Route Builder** | `src/config/routes.ts` | `routes.venue(slug)` |
| **Home Page** | `src/app/page.tsx` | `/` route |
| **Login Page** | `src/app/(auth)/login/page.tsx` | `/login` route |

## 🎯 Decision Tree

```
New Code?
│
├─ Is it a page/route?
│  └─→ src/app/
│
├─ Is it reusable across features?
│  └─→ src/components/
│
├─ Is it specific to one feature?
│  └─→ src/features/[feature]/
│
├─ Is it a pure function?
│  └─→ src/lib/
│
├─ Is it a React hook?
│  └─→ src/hooks/ (shared) or features/[feature]/hooks/ (specific)
│
└─ Is it a type?
   └─→ src/types/ (shared) or features/[feature]/types/ (specific)
```

## 🔗 Import Patterns

```typescript
// ✅ Correct
import { Button } from "@/components/ui/button";
import { HeroSection } from "@/features/home/components/HeroSection";
import { formatCurrency } from "@/lib/formatCurrency";
import { routes } from "@/config/routes";

// ❌ Wrong
import { Button } from "../../components/ui/button";
import { HeroSection } from "@/features/home/components/HeroSection"; // In Button.tsx
```

## 📋 Component Checklist

- [ ] Is it generic/reusable? → `components/`
- [ ] Is it feature-specific? → `features/[feature]/components/`
- [ ] Does it use business logic? → `features/`
- [ ] Is it a route? → `app/`
- [ ] Uses `@/` for imports? ✅
- [ ] Named export (not default)? ✅




