# Payment Methods Debugging Guide

## Issue: Google Pay and Link Not Showing in Payment Element

### Frontend Configuration ✅
The frontend is correctly configured with:
- `wallets: { applePay: "auto", googlePay: "auto" }`
- Business name set for better wallet support
- Proper event listeners for debugging

### What to Check on Backend

#### 1. Payment Intent Creation
When creating the Payment Intent via `/customer/upfront-payment-info`, ensure:

**For Payment Intents:**
```javascript
// The Payment Intent should include these payment method types:
payment_method_types: ['card', 'link']  // Link must be explicitly included

// For Stripe Connect accounts:
payment_method_types: ['card', 'link'],
on_behalf_of: connectedAccountId,  // The salon's Stripe account
transfer_data: {
  destination: connectedAccountId,
}
```

**For Setup Intents (when deposit is 0):**
```javascript
// Setup Intent should also include link:
payment_method_types: ['card', 'link']
```

#### 2. Stripe Connect Account Settings
The connected account (salon's Stripe account) must have:
- **Link enabled**: Go to Stripe Dashboard → Settings → Payment methods → Enable "Link"
- **Google Pay enabled**: Settings → Payment methods → Enable "Google Pay"
- **Apple Pay enabled**: Settings → Payment methods → Enable "Apple Pay"

#### 3. Country and Currency Support
- **Link**: Available in US, UK, and other supported countries
- **Google Pay**: Available in many countries, but requires HTTPS
- **Apple Pay**: Available on Apple devices with Safari

#### 4. Testing Checklist

1. **Check Browser Console**: Look for logs showing:
   - Payment Intent/Setup Intent details
   - Available payment method types
   - Payment Element change events

2. **Verify HTTPS**: Google Pay and Apple Pay require HTTPS (except localhost)

3. **Check Stripe Dashboard**:
   - Go to the connected account's dashboard
   - Verify Link and Google Pay are enabled
   - Check if there are any restrictions

4. **Test Payment Intent**:
   - Create a test Payment Intent in Stripe Dashboard
   - Check what payment methods are available
   - Compare with what your backend creates

### Debugging Steps

1. **Open browser console** when making a booking
2. **Look for these logs**:
   - "Payment Intent details:" or "Setup Intent details:"
   - "Payment Element change event:"
   - Check the `payment_method_types` array

3. **If `payment_method_types` doesn't include `link`**:
   - Update backend to include `link` in the Payment Intent creation

4. **If Link is in the array but not showing**:
   - Check if Link is enabled on the connected account
   - Verify country/currency support
   - Check if there are any account restrictions

### Expected Console Output

When working correctly, you should see:
```
Payment Element created with configuration: { hasWallets: true, ... }
Payment Intent details: { payment_method_types: ['card', 'link'], ... }
Payment Element change event: { value: { type: 'card' }, ... }
```

### Backend API Requirements

**For `/customer/upfront-payment-info`:**
- Must create Payment Intent with `payment_method_types: ['card', 'link']`
- Must use Stripe Connect with proper `on_behalf_of` or `transfer_data`

**For `/customer/create-setup-intent`:**
- Must create Setup Intent with `payment_method_types: ['card', 'link']`
- Must use Stripe Connect with proper account connection

### Contact Backend Team

If payment methods are still not showing after checking the above:
1. Share the console logs showing Payment Intent details
2. Verify the connected account has Link/Google Pay enabled
3. Check if the Payment Intent is being created with the correct payment method types

