# Project Architecture

This document describes the architecture and structure of the Fresha clone project.

## Directory Structure

```
src/
├── app/                    # Next.js App Router routes
│   ├── layout.tsx          # Root layout with Header/Footer
│   ├── page.tsx            # Home page
│   ├── (auth)/             # Auth route group
│   │   ├── login/
│   │   └── signup/
│   ├── (customer)/         # Customer route group
│   │   └── profile/
│   ├── (marketing)/        # Marketing route group
│   │   └── business/
│   └── (venues)/           # Venues route group
│       └── [venueSlug]/
│
├── components/             # Reusable UI components
│   ├── layout/            # Layout components
│   │   ├── Container.tsx
│   │   └── PageSection.tsx
│   ├── navigation/        # Navigation components
│   │   ├── Header.tsx
│   │   └── Footer.tsx
│   ├── ui/                # Base UI components (design system)
│   │   ├── button.tsx
│   │   └── card.tsx
│   ├── carousel/          # Carousel components
│   │   └── HorizontalCarousel.tsx
│   └── data-display/      # Data display components
│       └── StarRating.tsx
│
├── features/              # Feature-based components
│   └── home/              # Home page features
│       ├── components/
│       │   ├── HeroSection.tsx
│       │   ├── VenueCard.tsx
│       │   ├── RecommendedSection.tsx
│       │   ├── NewToFreshaSection.tsx
│       │   ├── TrendingSection.tsx
│       │   ├── AppDownloadSection.tsx
│       │   ├── ReviewsSection.tsx
│       │   ├── TopStatsSection.tsx
│       │   ├── BusinessPromoSection.tsx
│       │   └── CityDirectorySection.tsx
│       └── hooks/
│           └── useHomepageData.ts
│
├── lib/                   # Utility functions
│   ├── classNames.ts
│   ├── formatCurrency.ts
│   ├── formatDate.ts
│   └── formatTime.ts
│
├── hooks/                 # Shared React hooks
│   ├── useDebounce.ts
│   ├── useMediaQuery.ts
│   └── useToggle.ts
│
├── config/                # Configuration files
│   ├── constants.ts
│   └── routes.ts
│
├── types/                 # TypeScript types
│   ├── common.ts
│   └── venue.ts
│
├── store/                 # State management (Zustand)
│   └── useStore.ts
│
└── styles/                # Global styles
    └── globals.css
```

## Architecture Principles

### 1. Feature-Based Organization
- Features are self-contained modules in `src/features/`
- Each feature has its own components, hooks, and API functions
- Features don't depend on other features directly

### 2. Component Hierarchy
- **Layout Components**: Structure and containers (`Container`, `PageSection`)
- **UI Components**: Reusable design system primitives (`Button`, `Card`)
- **Feature Components**: Domain-specific components (`HeroSection`, `VenueCard`)
- **Navigation Components**: Global navigation (`Header`, `Footer`)

### 3. Route Groups
Next.js route groups `(auth)`, `(customer)`, `(marketing)`, `(venues)` organize routes without affecting URLs:
- Clean URLs (no `/auth/login`, just `/login`)
- Shared layouts per route group
- Logical grouping of related pages

### 4. State Management
- Zustand for client-side state
- Separate stores for different domains (search, carousels)
- Easy to extend with new stores

### 5. Type Safety
- TypeScript throughout
- Shared types in `src/types/`
- Feature-specific types in feature directories

## Key Features

### Home Page Sections
1. **HeroSection**: Main search bar and CTA
2. **RecommendedSection**: Carousel of recommended venues
3. **NewToFreshaSection**: New venues carousel
4. **TrendingSection**: Trending venues carousel
5. **AppDownloadSection**: App download with phone mockups
6. **ReviewsSection**: Customer testimonials
7. **TopStatsSection**: Statistics and metrics
8. **BusinessPromoSection**: Business-focused marketing
9. **CityDirectorySection**: City browsing directory

### Reusable Components
- `Container`: Max-width container with padding
- `PageSection`: Section wrapper with background options
- `Button`: Styled button with variants
- `Card`: Card component with hover effects
- `HorizontalCarousel`: Scrollable carousel with arrows
- `StarRating`: Star rating display component

## Next Steps

### To Complete the Architecture:

1. **Venues Feature** (`src/features/venues/`)
   - Venue detail page components
   - Service listings
   - Team/staff sections
   - Reviews integration

2. **Booking Feature** (`src/features/booking/`)
   - Booking flow components
   - Step-by-step booking process
   - Payment integration
   - Booking state management

3. **Auth Feature** (`src/features/auth/`)
   - Login/signup forms
   - Social auth integration
   - Session management

4. **Profile Feature** (`src/features/profile/`)
   - User profile management
   - Appointment history
   - Favorites
   - Settings

5. **Additional UI Components**
   - Input fields
   - Select dropdowns
   - Modals/Dialogs
   - Toast notifications
   - Loading states

6. **API Layer**
   - API route handlers
   - Data fetching utilities
   - Error handling
   - Caching strategies

## Development Guidelines

1. **Component Creation**: Place feature-specific components in `features/`, reusable ones in `components/`
2. **State Management**: Use Zustand stores for client state, React state for component-local state
3. **Styling**: Use Tailwind CSS with utility classes, maintain consistency with design system
4. **Type Safety**: Always define TypeScript types, avoid `any`
5. **File Naming**: Use PascalCase for components, camelCase for utilities
6. **Imports**: Use `@/` alias for all imports from `src/`




