/**
 * Authentication Form Validation
 * 
 * Validation functions for auth forms (signup, login, etc.)
 */

export interface ValidationErrors {
  firstName?: string;
  lastName?: string;
  email?: string;
  password?: string;
  phoneNumber?: string;
  agreedToTerms?: string;
}

export function validateFirstName(firstName: string): string | undefined {
  if (!firstName || firstName.trim().length === 0) {
    return "First name is required";
  }
  if (firstName.trim().length < 2) {
    return "First name must be at least 2 characters";
  }
  if (firstName.trim().length > 50) {
    return "First name must be less than 50 characters";
  }
  return undefined;
}

export function validateLastName(lastName: string): string | undefined {
  if (!lastName || lastName.trim().length === 0) {
    return "Last name is required";
  }
  if (lastName.trim().length < 2) {
    return "Last name must be at least 2 characters";
  }
  if (lastName.trim().length > 50) {
    return "Last name must be less than 50 characters";
  }
  return undefined;
}

export function validateEmail(email: string): string | undefined {
  if (!email || email.trim().length === 0) {
    return "Email address is required";
  }
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if (!emailRegex.test(email.trim())) {
    return "Please enter a valid email address";
  }
  return undefined;
}

export function validatePassword(password: string): string | undefined {
  if (!password || password.length === 0) {
    return "Password is required";
  }
  if (password.length < 8) {
    return "Password must be at least 8 characters";
  }
  if (password.length > 128) {
    return "Password must be less than 128 characters";
  }
  // Check for at least one letter and one number
  const hasLetter = /[a-zA-Z]/.test(password);
  const hasNumber = /[0-9]/.test(password);
  if (!hasLetter || !hasNumber) {
    return "Password must contain at least one letter and one number";
  }
  return undefined;
}

export function validatePhoneNumber(phoneNumber: string): string | undefined {
  if (!phoneNumber || phoneNumber.trim().length === 0) {
    return "Mobile number is required";
  }
  // Remove spaces, dashes, and parentheses for validation
  const cleaned = phoneNumber.replace(/[\s\-\(\)]/g, "");
  if (!/^\d+$/.test(cleaned)) {
    return "Phone number must contain only digits";
  }
  if (cleaned.length < 7) {
    return "Phone number is too short";
  }
  if (cleaned.length > 15) {
    return "Phone number is too long";
  }
  return undefined;
}

export function validateTermsAgreement(agreed: boolean): string | undefined {
  if (!agreed) {
    return "You must agree to the terms and conditions";
  }
  return undefined;
}

export function validateSignupForm(data: {
  firstName: string;
  lastName: string;
  email: string;
  password: string;
  phoneNumber: string;
  agreedToTerms: boolean;
}): ValidationErrors {
  const errors: ValidationErrors = {};

  const firstNameError = validateFirstName(data.firstName);
  if (firstNameError) errors.firstName = firstNameError;

  const lastNameError = validateLastName(data.lastName);
  if (lastNameError) errors.lastName = lastNameError;

  const emailError = validateEmail(data.email);
  if (emailError) errors.email = emailError;

  const passwordError = validatePassword(data.password);
  if (passwordError) errors.password = passwordError;

  const phoneError = validatePhoneNumber(data.phoneNumber);
  if (phoneError) errors.phoneNumber = phoneError;

  const termsError = validateTermsAgreement(data.agreedToTerms);
  if (termsError) errors.agreedToTerms = termsError;

  return errors;
}
