/**
 * Zod Schemas for Salon Detail API
 * All fields are optional to handle partial data
 */

import { z } from "zod";

export const AddressDBSchema = z.object({
  id: z.number().optional(),
  streetAddress: z.string().nullable().optional(),
  city: z.string().nullable().optional(),
  district: z.string().nullable().optional(),
  province: z.string().nullable().optional(),
  country: z.string().nullable().optional(),
  lat: z.string().nullable().optional(),
  lng: z.string().nullable().optional(),
  building: z.string().nullable().optional(),
  postalCode: z.string().nullable().optional(),
});

export const ServiceSchema = z.object({
  id: z.number().optional(),
  serviceName: z.string().nullable().optional(),
  description: z.string().nullable().optional(),
  price: z.string().nullable().optional(),
  duration: z.number().nullable().optional(),
  discount: z.number().nullable().optional(),
});

export const CategorySchema = z.object({
  id: z.number().optional(),
  categoryName: z.string().nullable().optional(),
  color: z.string().nullable().optional(),
  services: z.array(ServiceSchema).optional(),
});

export const SalonImageSchema = z.object({
  imageUrl: z.string().nullable().optional(),
});

export const TimeSchema = z.object({
  status: z.boolean().optional(),
  day: z.string().optional(),
  openingTime: z.string().nullable().optional(),
  closingTime: z.string().nullable().optional(),
});

export const EmployeeUserSchema = z.object({
  id: z.number().optional(),
  firstName: z.string().nullable().optional(),
  lastName: z.string().nullable().optional(),
  image: z.string().nullable().optional(),
});

export const EmployeeSchema = z.object({
  id: z.number().optional(),
  position: z.string().nullable().optional(),
  employeeAverageRating: z.string().nullable().optional(),
  user: EmployeeUserSchema.optional(),
});

export const BookingUserSchema = z.object({
  firstName: z.string().nullable().optional(),
  lastName: z.string().nullable().optional(),
  image: z.string().nullable().optional(),
});

export const BookingSchema = z.object({
  id: z.number().optional(),
  feedback: z.string().nullable().optional(),
  createdAt: z.string().nullable().optional(),
  vatAmount: z.string().nullable().optional(),
  averageRatingPerBooking: z.string().nullable().optional(),
  user: BookingUserSchema.optional(),
});

export const SalonDetailSchema = z.object({
  id: z.number().optional(),
  salonName: z.string().nullable().optional(),
  metadata: z.string().nullable().optional(),
  coverImage: z.string().nullable().optional(),
  description: z.string().nullable().optional(),
  onlinePayments: z.boolean().nullable().optional(),
  isOnline: z.boolean().nullable().optional(),
  userId: z.number().nullable().optional(),
  isExternalEmployee: z.boolean().nullable().optional(),
  isVATRegistered: z.boolean().nullable().optional(),
  vatType: z.string().nullable().optional(),
  stripeAccountConnected: z.number().nullable().optional(),
  platformFee: z
    .any()
    .transform((val) => {
      if (val === null || val === undefined || val === "" || val === "null") {
        return null;
      }
      if (typeof val === "number") {
        return val;
      }
      if (typeof val === "string") {
        const parsed = parseFloat(val);
        return isNaN(parsed) ? null : parsed;
      }
      return null;
    })
    .pipe(z.number().nullable())
    .optional(),
  salonAverageRating: z
    .any()
    .transform((val) => {
      if (val === null || val === undefined || val === "" || val === "null") {
        return null;
      }
      if (typeof val === "number") {
        return val;
      }
      if (typeof val === "string") {
        const parsed = parseFloat(val);
        return isNaN(parsed) ? null : parsed;
      }
      return null;
    })
    .pipe(z.number().nullable())
    .optional(),
  ratingCount: z.number().nullable().optional(),
  socialLinks: z.array(z.unknown()).optional(),
  addressDB: AddressDBSchema.optional(),
  categories: z.array(CategorySchema).optional(),
  salonImages: z.array(SalonImageSchema).optional(),
  times: z.array(TimeSchema).optional(),
  employees: z.array(EmployeeSchema).optional(),
  bookings: z.array(BookingSchema).optional(),
});

export const CancellationPolicySchema = z.object({
  id: z.number().optional(),
  hoursBeforeBooking: z.string().nullable().optional(),
  refundPercentage: z.number().nullable().optional(),
  status: z.boolean().nullable().optional(),
  version: z.number().nullable().optional(),
});

export const NoShowPolicySchema = z.object({
  id: z.number().optional(),
  refundPercentage: z.number().nullable().optional(),
  status: z.boolean().nullable().optional(),
  version: z.number().nullable().optional(),
});

export const ReschedulePolicySchema = z.object({
  id: z.number().optional(),
  hoursBeforeBooking: z.number().nullable().optional(),
  count: z.number().nullable().optional(),
  version: z.number().nullable().optional(),
  status: z.boolean().nullable().optional(),
});

export const PaymentPolicySchema = z.object({
  id: z.number().optional(),
  percentage: z.number().nullable().optional(),
  status: z.boolean().nullable().optional(),
  version: z.number().nullable().optional(),
  createdAt: z.string().nullable().optional(),
  updatedAt: z.string().nullable().optional(),
  salonDetailId: z.number().nullable().optional(),
});

export const PoliciesSchema = z.object({
  id: z.number().optional(),
  salonName: z.string().nullable().optional(),
  cancellationPolicies: z.array(CancellationPolicySchema).optional(),
  noShowPolicies: z.array(NoShowPolicySchema).optional(),
  reschedulePolicies: z.array(ReschedulePolicySchema).optional(),
  paymentPolicies: z.array(PaymentPolicySchema).optional(),
});

export const SalonDetailResponseSchema = z.object({
  announcements: z.array(z.unknown()).optional(),
  customerStatus: z.string().nullable().optional(),
  like: z
    .union([z.boolean(), z.string(), z.number(), z.null(), z.undefined()])
    .transform((val) => {
      // Handle null/undefined/empty
      if (val === null || val === undefined || val === "" || val === "null") {
        return false;
      }
      // Handle boolean - return as-is
      if (typeof val === "boolean") {
        return val;
      }
      // Handle string
      if (typeof val === "string") {
        const lower = val.toLowerCase().trim();
        return lower === "true" || lower === "1" || lower === "yes";
      }
      // Handle number
      if (typeof val === "number") {
        return val === 1 || val > 0;
      }
      // Default to false for any other case
      return false;
    })
    .pipe(z.boolean())
    .default(false),
  detail: SalonDetailSchema.optional(),
  policies: PoliciesSchema.optional(),
  currencyUnit: z.string().nullable().optional(),
  feedbackCount: z.number().nullable().optional(),
});

export type AddressDB = z.infer<typeof AddressDBSchema>;
export type Service = z.infer<typeof ServiceSchema>;
export type Category = z.infer<typeof CategorySchema>;
export type SalonImage = z.infer<typeof SalonImageSchema>;
export type Time = z.infer<typeof TimeSchema>;
export type Employee = z.infer<typeof EmployeeSchema>;
export type Booking = z.infer<typeof BookingSchema>;
export type SalonDetail = z.infer<typeof SalonDetailSchema>;
export type Policies = z.infer<typeof PoliciesSchema>;
export type SalonDetailResponse = z.infer<typeof SalonDetailResponseSchema>;
