/**
 * Zod Schemas for Home page API
 */

import { z } from "zod";

export const AddressDBSchema = z.object({
  id: z.number(),
  streetAddress: z.string(),
  city: z.string(),
  district: z.string(),
  province: z.string(),
  country: z.string(),
  lat: z.string(),
  lng: z.string(),
  building: z.string(),
  postalCode: z.string(),
  zoneId: z.number().nullable(),
});

export const HomeServiceSchema = z.object({
  id: z.number(),
  serviceName: z.string(),
  price: z.string(),
  discount: z.number(),
});

export const HomeSalonSchema = z.object({
  id: z.number(),
  salonName: z.string(),
  coverImage: z.string().nullable(),
  isExternalEmployee: z.boolean(),
  isOnline: z.boolean(),
  // API can return rating as string or number, coerce to number
  salonAverageRating: z
    .any()
    .transform((val) => {
      if (val === null || val === undefined || val === "" || val === "null") {
        return null;
      }
      if (typeof val === "number") {
        return val;
      }
      if (typeof val === "string") {
        const parsed = parseFloat(val);
        return isNaN(parsed) ? null : parsed;
      }
      return null;
    })
    .pipe(z.number().nullable()),
  ratingCount: z.number(),
  distance: z.number(),
  addressDB: AddressDBSchema,
  services: z.array(HomeServiceSchema),
  user: z.object({
    id: z.number(),
  }),
  like: z.boolean(),
  averageDiscount: z.number(),
});

export const GetHomeVenuesRequestSchema = z.object({
  targetLat: z.number(),
  targetLng: z.number(),
  rangeInKm: z.number().min(0),
  serviceType: z.string().nullable().optional(),
});

export const GetHomeVenuesResponseSchema = z.object({
  slons: z.array(HomeSalonSchema), // Note: API uses "slons" (typo)
  appointments: z.array(z.unknown()).optional(), // Define appointment schema if needed
  currencyUnit: z.string(),
  distanceUnit: z.string(),
  highestPriceAcrossAllSalons: z.string(),
  lowestPriceAcrossAllSalons: z.string(),
});

export type AddressDB = z.infer<typeof AddressDBSchema>;
export type HomeService = z.infer<typeof HomeServiceSchema>;
export type HomeSalon = z.infer<typeof HomeSalonSchema>;
export type GetHomeVenuesRequest = z.infer<typeof GetHomeVenuesRequestSchema>;
export type GetHomeVenuesResponse = z.infer<typeof GetHomeVenuesResponseSchema>;
