/**
 * Server-side Authentication Helpers
 * Tokens are read-only on the server from cookies
 * ONLY use this in Server Components, Server Actions, or Route Handlers
 */

import "server-only";
import { cookies } from "next/headers";

/**
 * Get access token from server-side cookies
 * Only use this in Server Components, Server Actions, or Route Handlers
 */
export async function getServerToken(): Promise<string | null> {
  const cookieStore = await cookies();
  return cookieStore.get("access_token")?.value || null;
}

/**
 * Get user data from server-side cookies
 */
export async function getServerUserData(): Promise<any | null> {
  const cookieStore = await cookies();
  const userData = cookieStore.get("user_data")?.value;
  return userData ? JSON.parse(userData) : null;
}

/**
 * Check if user is authenticated on server
 */
export async function isServerAuthenticated(): Promise<boolean> {
  const token = await getServerToken();
  return !!token;
}

/**
 * Set token in cookies (server-side)
 * Use this in Server Actions or Route Handlers after login
 */
export async function setServerToken(
  token: string,
  userData?: any
): Promise<void> {
  const cookieStore = await cookies();

  cookieStore.set("access_token", token, {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "lax",
    maxAge: 60 * 60 * 24 * 7, // 7 days
  });

  if (userData) {
    cookieStore.set("user_data", JSON.stringify(userData), {
      httpOnly: false, // Allow client-side access for hydration
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax",
      maxAge: 60 * 60 * 24 * 7,
    });
  }
}

/**
 * Clear server-side auth cookies
 */
export async function clearServerAuth(): Promise<void> {
  const cookieStore = await cookies();
  cookieStore.delete("access_token");
  cookieStore.delete("user_data");
}
