/**
 * Get Venue API
 * Fetches a single venue by slug
 */

import { clientFetcher } from "@/lib/api/fetcher-client";
import { VenueSchema, type Venue } from "@/lib/schemas/venue";

export interface GetVenueParams {
  slug: string;
}

/**
 * Server-side: Get venue by slug
 * Use in Server Components, Server Actions, Route Handlers
 * Dynamically imports server fetcher to avoid bundling in client code
 */
export async function getVenueServer(
  params: GetVenueParams
): Promise<Venue> {
  // Dynamically import server fetcher to avoid bundling in client
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const data = await serverFetcher<unknown>(
    `/venues/${params.slug}`,
    {
      requireAuth: false, // Public endpoint
      method: "GET",
    }
  );

  return VenueSchema.parse(data);
}

/**
 * Client-side: Get venue by slug
 * Use in Client Components, React Query hooks
 */
export async function getVenueClient(
  params: GetVenueParams
): Promise<Venue> {
  const data = await clientFetcher<unknown>(
    `/venues/${params.slug}`,
    {
      requireAuth: false,
      method: "GET",
    }
  );

  return VenueSchema.parse(data);
}
