"use client";

import { useState, useEffect, useRef } from "react";
import { Search, X, MapPin } from "lucide-react";
import { getServiceTypesClient, ServiceType } from "../api/getServiceTypes-client";
import { getHomeVenuesClient } from "@/features/home/api/getHomeVenues-client";
import { transformSalonToVenue } from "@/features/home/utils/transformSalonToVenue";
import { Venue } from "@/types/venue";
import Image from "next/image";
import { API_BASE_URL } from "@/config/env";
import { useRouter } from "next/navigation";

interface ServiceTypesDropdownProps {
  isOpen: boolean;
  onClose: () => void;
  onSelect: (serviceType: ServiceType | null) => void;
  selectedService: ServiceType | null;
  location?: { lat: number; lng: number } | null;
}

export function ServiceTypesDropdown({
  isOpen,
  onClose,
  onSelect,
  selectedService,
  location,
}: ServiceTypesDropdownProps) {
  const router = useRouter();
  const [serviceTypes, setServiceTypes] = useState<ServiceType[]>([]);
  const [searchQuery, setSearchQuery] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [venues, setVenues] = useState<Venue[]>([]);
  const [isLoadingVenues, setIsLoadingVenues] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (isOpen) {
      fetchServiceTypes();
    }
  }, [isOpen]);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        dropdownRef.current &&
        !dropdownRef.current.contains(event.target as Node)
      ) {
        onClose();
      }
    };

    if (isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
    }

    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, [isOpen, onClose]);

  const fetchServiceTypes = async () => {
    setIsLoading(true);
    try {
      const data = await getServiceTypesClient();
      // Sort by serviceOrder
      const sorted = [...data].sort((a, b) => a.serviceOrder - b.serviceOrder);
      setServiceTypes(sorted);
    } catch (error) {
      console.error("Failed to fetch service types:", error);
      setServiceTypes([]);
    } finally {
      setIsLoading(false);
    }
  };

  // Fetch venues when user types in search
  useEffect(() => {
    if (searchQuery.trim().length > 0 && location) {
      const fetchVenues = async () => {
        setIsLoadingVenues(true);
        try {
          const response = await getHomeVenuesClient({
            targetLat: location.lat,
            targetLng: location.lng,
            rangeInKm: 10,
            serviceType: null, // Search all venues regardless of service type
          });
          const transformedVenues = response.slons.map(transformSalonToVenue);
          // Filter venues by search query
          const filtered = transformedVenues.filter((venue) =>
            venue.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
            venue.location.toLowerCase().includes(searchQuery.toLowerCase()) ||
            venue.address.toLowerCase().includes(searchQuery.toLowerCase())
          );
          setVenues(filtered);
        } catch (error) {
          console.error("Failed to fetch venues:", error);
          setVenues([]);
        } finally {
          setIsLoadingVenues(false);
        }
      };

      // Debounce the search
      const timeoutId = setTimeout(() => {
        fetchVenues();
      }, 300);

      return () => clearTimeout(timeoutId);
    } else {
      setVenues([]);
    }
  }, [searchQuery, location]);

  const filteredServices = serviceTypes.filter((service) =>
    service.typeName.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const handleVenueClick = (venue: Venue) => {
    router.push(`/saloons/${venue.id}`);
    onClose();
  };

  const getImageUrl = (imagePath: string) => {
    if (imagePath.startsWith("http://") || imagePath.startsWith("https://")) {
      return imagePath;
    }
    return imagePath.startsWith("/")
      ? `${API_BASE_URL}${imagePath}`
      : `${API_BASE_URL}/${imagePath}`;
  };

  if (!isOpen) return null;

  return (
    <div
      ref={dropdownRef}
      className="absolute top-full left-0 mt-2 bg-white rounded-xl shadow-lg border border-gray-200 z-50 max-h-96 overflow-hidden flex flex-col"
      style={{ width: "312px" }}
    >
      {/* Search Input */}
      <div className="p-4 border-b border-gray-200">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
          <input
            type="text"
            placeholder="Search services..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-500"
          />
        </div>
      </div>

      {/* Service Types List */}
      <div className="overflow-y-auto flex-1">
        {isLoading ? (
          <div className="p-8 text-center text-gray-500">Loading...</div>
        ) : (
          <>
            {/* All treatments option - only show when no search query */}
            {!searchQuery && (
              <button
                onClick={() => {
                  onSelect(null);
                  onClose();
                }}
                className={`w-full px-4 py-3 text-left hover:bg-gray-50 flex items-center gap-3 ${
                  !selectedService ? "bg-purple-50" : ""
                }`}
              >
                <div className="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center border border-gray-200">
                  <div className="grid grid-cols-2 gap-0.5">
                    <div className="w-1.5 h-1.5 bg-gray-400 rounded-sm"></div>
                    <div className="w-1.5 h-1.5 bg-gray-400 rounded-sm"></div>
                    <div className="w-1.5 h-1.5 bg-gray-400 rounded-sm"></div>
                    <div className="w-1.5 h-1.5 bg-gray-400 rounded-sm"></div>
                  </div>
                </div>
                <span className="body-text">All treatments</span>
              </button>
            )}

            {/* Top categories heading */}
            {!searchQuery && (
              <div className="px-4 py-2">
                <h3 className="heading-6 text-gray-900">Top categories</h3>
              </div>
            )}

            {/* Service types */}
            {filteredServices.length > 0 ? (
              filteredServices.map((service) => (
                <button
                  key={service.id}
                  onClick={() => {
                    onSelect(service);
                    onClose();
                  }}
                  className={`w-full px-4 py-3 text-left hover:bg-gray-50 flex items-center gap-3 ${
                    selectedService?.id === service.id ? "bg-purple-50" : ""
                  }`}
                >
                  <div className="w-10 h-10 rounded-lg overflow-hidden flex-shrink-0 border border-gray-200">
                    <Image
                      src={getImageUrl(service.image)}
                      alt={service.typeName}
                      width={40}
                      height={40}
                      className="w-full h-full object-cover"
                    />
                  </div>
                  <span className="body-text">{service.typeName}</span>
                </button>
              ))
            ) : searchQuery.trim().length > 0 ? (
              <div className="p-4 text-center text-gray-500 text-sm">
                No services found
              </div>
            ) : null}

            {/* Venues section - shown when search query exists */}
            {searchQuery.trim().length > 0 && (
              <>
                {filteredServices.length > 0 && (
                  <div className="border-t border-gray-200 my-2"></div>
                )}
                <div className="px-4 py-2">
                  <h3 className="heading-6 text-gray-900">Venues</h3>
                </div>
                {isLoadingVenues ? (
                  <div className="p-4 text-center text-gray-500 text-sm">
                    Searching venues...
                  </div>
                ) : venues.length === 0 ? (
                  <div className="p-4 text-center text-gray-500 text-sm">
                    No venues found
                  </div>
                ) : (
                  venues.map((venue) => (
                    <button
                      key={venue.id}
                      onClick={() => handleVenueClick(venue)}
                      className="w-full px-4 py-3 text-left hover:bg-gray-50 flex items-center gap-3"
                    >
                      <div className="w-10 h-10 rounded-lg overflow-hidden flex-shrink-0 border border-gray-200 bg-gray-100 flex items-center justify-center">
                        {venue.image && venue.image !== "/placeholder-venue" ? (
                          <Image
                            src={venue.image}
                            alt={venue.name}
                            width={40}
                            height={40}
                            className="w-full h-full object-cover"
                          />
                        ) : (
                          <MapPin className="w-5 h-5 text-gray-400" />
                        )}
                      </div>
                      <div className="flex-1 min-w-0">
                        <div className="body-text truncate">{venue.name}</div>
                        <div className="text-xs text-gray-500 truncate">
                          {venue.location}
                        </div>
                      </div>
                    </button>
                  ))
                )}
              </>
            )}
          </>
        )}
      </div>
    </div>
  );
}

