"use client";

import { useState, useEffect, useCallback } from "react";
import { useRouter } from "next/navigation";
import { Search, MapPin, Filter, X, Heart, Star } from "lucide-react";
import { VenueCard } from "@/features/home/components/VenueCard";
import { searchSalonsClient } from "../api/searchSalons-client";
import { transformSalonToVenue } from "@/features/home/utils/transformSalonToVenue";
import { Venue } from "@/types/venue";
import { googleApiKey } from "@/config/env";
import { HomeSalon } from "@/lib/schemas/home";
import { Dialog } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";

interface SearchResultsPageProps {
  searchParams: {
    lat?: string;
    lng?: string;
    serviceType?: string;
    day?: string;
    time?: string;
    price?: string;
  };
}

export function SearchResultsPage({ searchParams }: SearchResultsPageProps) {
  const router = useRouter();
  const [venues, setVenues] = useState<Venue[]>([]);
  const [salons, setSalons] = useState<HomeSalon[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [mapVisible, setMapVisible] = useState(false);
  const [map, setMap] = useState<google.maps.Map | null>(null);
  const [mobileMap, setMobileMap] = useState<google.maps.Map | null>(null);
  const [markers, setMarkers] = useState<google.maps.Marker[]>([]);
  const [selectedVenue, setSelectedVenue] = useState<Venue | null>(null);
  const [isMapLoading, setIsMapLoading] = useState(true);
  const [isMobileMapLoading, setIsMobileMapLoading] = useState(false);
  const [isDesktopMapInitialized, setIsDesktopMapInitialized] = useState(false);
  const [isMobileMapInitialized, setIsMobileMapInitialized] = useState(false);

  // Store original unfiltered data
  const [originalSalons, setOriginalSalons] = useState<HomeSalon[]>([]);
  const [originalVenues, setOriginalVenues] = useState<Venue[]>([]);

  // Filter modal state
  const [isFilterModalOpen, setIsFilterModalOpen] = useState(false);
  const [sortBy, setSortBy] = useState<"best-match" | "nearest" | "top-rated">(
    "best-match"
  );
  const [maxPrice, setMaxPrice] = useState(700);
  const [minRating, setMinRating] = useState<number | null>(null);

  // Applied filter state (what's actually being used)
  const [appliedSortBy, setAppliedSortBy] = useState<
    "best-match" | "nearest" | "top-rated"
  >("best-match");
  const [appliedMaxPrice, setAppliedMaxPrice] = useState(700);
  const [appliedMinRating, setAppliedMinRating] = useState<number | null>(null);

  // Reset to applied values when modal opens
  useEffect(() => {
    if (isFilterModalOpen) {
      setSortBy(appliedSortBy);
      setMaxPrice(appliedMaxPrice);
      setMinRating(appliedMinRating);
    }
  }, [isFilterModalOpen, appliedSortBy, appliedMaxPrice, appliedMinRating]);

  const lat = searchParams.lat ? parseFloat(searchParams.lat) : 31.462576;
  const lng = searchParams.lng ? parseFloat(searchParams.lng) : 74.247932;

  const fetchSearchResults = async () => {
    setIsLoading(true);
    try {
      const searchRequest = {
        targetLat: lat,
        targetLng: lng,
        rangeInKm: 500,
        day: searchParams.day,
        time: searchParams.time,
        price: searchParams.price ? parseFloat(searchParams.price) : undefined,
        serviceType: searchParams.serviceType
          ? parseInt(searchParams.serviceType)
          : undefined,
      };

      const response = await searchSalonsClient(searchRequest);
      console.log("Search API Response:", response);
      console.log("Salons:", response.slons);
      console.log("Salons count:", response.slons?.length || 0);

      if (response.slons && response.slons.length > 0) {
        // Store original unfiltered data
        setOriginalSalons(response.slons);
        const transformedVenues = response.slons.map(transformSalonToVenue);
        setOriginalVenues(transformedVenues);

        // Note: Filters will be applied automatically via useEffect when originalSalons/originalVenues are set
      } else {
        console.log("No salons in response");
        setSalons([]);
        setVenues([]);
        setOriginalSalons([]);
        setOriginalVenues([]);
      }
    } catch (error) {
      console.error("Failed to fetch search results:", error);
      setVenues([]);
      setSalons([]);
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    fetchSearchResults();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [
    searchParams.lat,
    searchParams.lng,
    searchParams.serviceType,
    searchParams.day,
    searchParams.time,
    searchParams.price,
  ]);

  // Listen for map toggle event from ProfileHeader
  useEffect(() => {
    const handleToggleMap = () => {
      setMapVisible((prev) => !prev);
    };
    window.addEventListener("toggleMap", handleToggleMap);
    return () => window.removeEventListener("toggleMap", handleToggleMap);
  }, []);

  // Listen for filter modal open event from ProfileHeader
  useEffect(() => {
    const handleOpenFilterModal = () => {
      setIsFilterModalOpen(true);
    };
    window.addEventListener("openFilterModal", handleOpenFilterModal);
    return () =>
      window.removeEventListener("openFilterModal", handleOpenFilterModal);
  }, []);

  useEffect(() => {
    if (venues.length > 0 && salons.length > 0) {
      // Clear existing markers
      markers.forEach((marker) => marker.setMap(null));
      const newMarkers: google.maps.Marker[] = [];

      // Create markers for desktop map
      if (map) {
        salons.forEach((salon, index) => {
          const venue = venues[index];
          const venueLat = parseFloat(salon.addressDB.lat);
          const venueLng = parseFloat(salon.addressDB.lng);

          const marker = new google.maps.Marker({
            position: { lat: venueLat, lng: venueLng },
            map: map,
            title: salon.salonName,
          });

          marker.addListener("click", () => {
            setSelectedVenue(venue);
          });

          newMarkers.push(marker);
        });

        // Fit bounds to show all markers on desktop map
        if (newMarkers.length > 0) {
          const bounds = new google.maps.LatLngBounds();
          newMarkers.forEach((marker) => {
            const position = marker.getPosition();
            if (position) bounds.extend(position);
          });
          map.fitBounds(bounds);
        }
      }

      // Create markers for mobile map
      if (mobileMap) {
        salons.forEach((salon, index) => {
          const venue = venues[index];
          const venueLat = parseFloat(salon.addressDB.lat);
          const venueLng = parseFloat(salon.addressDB.lng);

          const marker = new google.maps.Marker({
            position: { lat: venueLat, lng: venueLng },
            map: mobileMap,
            title: salon.salonName,
          });

          marker.addListener("click", () => {
            setSelectedVenue(venue);
          });
        });

        // Fit bounds to show all markers on mobile map
        if (salons.length > 0) {
          const bounds = new google.maps.LatLngBounds();
          salons.forEach((salon) => {
            bounds.extend({
              lat: parseFloat(salon.addressDB.lat),
              lng: parseFloat(salon.addressDB.lng),
            });
          });
          mobileMap.fitBounds(bounds);
        }
      }

      setMarkers(newMarkers);
    }
  }, [map, mobileMap, venues, salons]);

  useEffect(() => {
    setIsMapLoading(true);
    if (typeof window !== "undefined" && !window.google) {
      const script = document.createElement("script");
      script.src = `https://maps.googleapis.com/maps/api/js?key=${googleApiKey}&libraries=places`;
      script.async = true;
      script.defer = true;
      script.onload = () => {
        // Initialize desktop map after a delay to ensure DOM is ready
        setTimeout(() => {
          initializeDesktopMap();
        }, 500);
      };
      document.head.appendChild(script);
    } else if (window.google) {
      setTimeout(() => {
        initializeDesktopMap();
      }, 500);
    }
  }, []);

  useEffect(() => {
    // Handle mobile map visibility
    if (!mapVisible) {
      // Reset when hidden - the element is removed from DOM, so map instance becomes invalid
      setMobileMap(null);
      setIsMobileMapInitialized(false);
    } else if (mapVisible && window.google) {
      // Initialize when shown - wait for DOM element to be ready
      const checkAndInitialize = () => {
        const mapElement = document.getElementById("map-mobile");
        if (mapElement) {
          // Always initialize if we don't have a map or if initialization flag is false
          // This handles the case where the element was removed and recreated
          if (!mobileMap || !isMobileMapInitialized) {
            initializeMobileMap();
          }
        } else {
          // Element not ready yet, retry
          setTimeout(checkAndInitialize, 100);
        }
      };
      setTimeout(checkAndInitialize, 300);
    }
  }, [mapVisible, mobileMap, isMobileMapInitialized]);

  const initializeDesktopMap = () => {
    if (!window.google) {
      setIsMapLoading(false);
      return;
    }

    const mapElement = document.getElementById("map-desktop");
    if (mapElement && !isDesktopMapInitialized && !map) {
      try {
        const newMap = new google.maps.Map(mapElement, {
          center: { lat, lng },
          zoom: 12,
          mapTypeControl: false,
          fullscreenControl: false,
          streetViewControl: false,
          zoomControlOptions: {
            position: google.maps.ControlPosition.BOTTOM_RIGHT,
          },
        });
        setMap(newMap);
        setIsDesktopMapInitialized(true);

        // Trigger resize after a short delay to ensure proper rendering
        setTimeout(() => {
          if (window.google && newMap) {
            window.google.maps.event.trigger(newMap, "resize");
          }
        }, 300);

        setIsMapLoading(false);
      } catch (error) {
        console.error("Error initializing desktop map:", error);
        setIsMapLoading(false);
      }
    } else if (map && !isDesktopMapInitialized) {
      // Map already exists, mark as initialized
      setIsDesktopMapInitialized(true);
      setIsMapLoading(false);
    } else if (!mapElement) {
      // Retry if element not ready (might be hidden on mobile)
      const checkInterval = setInterval(() => {
        const element = document.getElementById("map-desktop");
        if (element) {
          clearInterval(checkInterval);
          setTimeout(() => {
            initializeDesktopMap();
          }, 100);
        }
      }, 200);
      // Stop checking after 5 seconds
      setTimeout(() => {
        clearInterval(checkInterval);
        setIsMapLoading(false);
      }, 5000);
    } else {
      setIsMapLoading(false);
    }
  };

  const initializeMobileMap = () => {
    if (!window.google) {
      setIsMobileMapLoading(false);
      return;
    }

    setIsMobileMapLoading(true);
    const mapElement = document.getElementById("map-mobile");
    if (mapElement) {
      try {
        // If map already exists, don't create a new one
        if (mobileMap) {
          // Trigger resize to ensure proper rendering
          setTimeout(() => {
            if (window.google && mobileMap) {
              window.google.maps.event.trigger(mobileMap, "resize");
              mobileMap.setCenter({ lat, lng });
            }
          }, 100);
          setIsMobileMapLoading(false);
          return;
        }

        // Create new map instance
        const newMap = new google.maps.Map(mapElement, {
          center: { lat, lng },
          zoom: 12,
          mapTypeControl: false,
          fullscreenControl: false,
          streetViewControl: false,
          zoomControlOptions: {
            position: google.maps.ControlPosition.BOTTOM_RIGHT,
          },
        });
        setMobileMap(newMap);
        setIsMobileMapInitialized(true);

        // Trigger resize after initialization
        setTimeout(() => {
          if (window.google && newMap) {
            window.google.maps.event.trigger(newMap, "resize");
          }
        }, 300);

        setIsMobileMapLoading(false);
      } catch (error) {
        console.error("Error initializing mobile map:", error);
        setIsMobileMapLoading(false);
      }
    } else {
      // Element not ready, retry
      setTimeout(() => {
        initializeMobileMap();
      }, 200);
    }
  };

  // Apply filters function
  const applyFilters = useCallback(
    (
      salonsToFilter: HomeSalon[],
      venuesToFilter: Venue[],
      sortType: "best-match" | "nearest" | "top-rated",
      priceFilter: number,
      ratingFilter: number | null
    ) => {
      console.log("Applying filters:", {
        sortType,
        priceFilter,
        ratingFilter,
        totalItems: salonsToFilter.length,
      });

      // Start with pairs to maintain salon-venue relationship
      let pairs = salonsToFilter.map((salon, index) => ({
        salon,
        venue: venuesToFilter[index],
      }));

      const initialCount = pairs.length;

      // Filter by price (check if any service price is within max price)
      // Only filter if priceFilter is less than 1000 (the max)
      if (priceFilter < 1000) {
        pairs = pairs.filter(({ salon }) => {
          // Check if any service has a price within the max price
          return salon.services.some((service) => {
            const servicePrice = parseFloat(service.price) || 0;
            return servicePrice <= priceFilter;
          });
        });
        console.log(
          `After price filter (≤£${priceFilter}): ${pairs.length} items (from ${initialCount})`
        );
      }

      // Filter by minimum rating
      if (ratingFilter !== null && ratingFilter > 0) {
        const beforeRatingFilter = pairs.length;
        pairs = pairs.filter(({ salon }) => {
          const rating = salon.salonAverageRating ?? 0;
          return rating >= ratingFilter;
        });
        console.log(
          `After rating filter (≥${ratingFilter}): ${pairs.length} items (from ${beforeRatingFilter})`
        );
      }

      // Sort based on sort type
      if (sortType === "nearest" || sortType === "best-match") {
        // Sort by distance (closest first) - like "Near You"
        pairs.sort((a, b) => a.salon.distance - b.salon.distance);
        console.log(
          `Sorted by ${sortType} (distance):`,
          pairs.slice(0, 3).map((p) => ({
            name: p.salon.salonName,
            distance: p.salon.distance,
          }))
        );
      } else if (sortType === "top-rated") {
        // Sort by rating (highest first) - like "Top Rated"
        // Filter out null ratings first, then sort
        const beforeTopRatedFilter = pairs.length;
        pairs = pairs.filter(({ salon }) => salon.salonAverageRating !== null);
        pairs.sort((a, b) => {
          const ratingA = a.salon.salonAverageRating ?? 0;
          const ratingB = b.salon.salonAverageRating ?? 0;
          return ratingB - ratingA; // Descending order (highest first)
        });
        console.log(
          `Sorted by top-rated: ${pairs.length} items (from ${beforeTopRatedFilter})`,
          pairs.slice(0, 3).map((p) => ({
            name: p.salon.salonName,
            rating: p.salon.salonAverageRating,
          }))
        );
      }

      // Extract salons and venues from pairs
      const filteredSalons = pairs.map(({ salon }) => salon);
      const filteredVenues = pairs.map(({ venue }) => venue);

      console.log(`Final filtered count: ${filteredSalons.length} items`);
      setSalons(filteredSalons);
      setVenues(filteredVenues);
    },
    []
  );

  // Apply filters when applied filter values change
  useEffect(() => {
    if (originalSalons.length > 0 && originalVenues.length > 0) {
      applyFilters(
        originalSalons,
        originalVenues,
        appliedSortBy,
        appliedMaxPrice,
        appliedMinRating
      );
    }
  }, [
    appliedSortBy,
    appliedMaxPrice,
    appliedMinRating,
    originalSalons,
    originalVenues,
    applyFilters,
  ]);

  const venueCount = venues.length;
  const venueCountText =
    venueCount >= 1000
      ? `${(venueCount / 1000).toFixed(1)}k+ venues nearby`
      : `${venueCount}+ venues nearby`;

  // Send venue count to ProfileHeader via custom event
  useEffect(() => {
    window.dispatchEvent(
      new CustomEvent("venueCountUpdate", { detail: venueCountText })
    );
  }, [venueCountText]);

  return (
    <div
      className="min-h-screen lg:pt-20"
      style={{ backgroundColor: "var(--surface)" }}
    >
      {/* Main Content */}
      <div className="flex flex-col sm:flex-row">
        {/* Left Panel - Venue List */}
        <div className="w-full sm:w-1/2 overflow-y-auto scrollbar-hide">
          <div className="px-4 sm:px-6 lg:px-8 py-6">
            {/* Venue Count and Action Buttons - Hidden on mobile, visible on tablet and desktop */}
            <div className="hidden sm:flex md:flex lg:flex flex-row items-center justify-between gap-4 mb-6">
              <p className="text-sm text-gray-600 font-medium">
                {venueCountText}
              </p>
              <div className="flex items-center gap-2">
                <button
                  onClick={() => setIsFilterModalOpen(true)}
                  className="px-4 py-2 border border-gray-300 rounded-full hover:bg-gray-50 flex items-center gap-2 text-sm font-medium transition-colors"
                >
                  <Filter className="w-4 h-4" />
                  <span>Filters</span>
                </button>
              </div>
            </div>
            {isLoading ? (
              <div className="text-center py-12">
                <p className="body-text text-gray-500">Loading venues...</p>
              </div>
            ) : venues.length === 0 ? (
              <div className="text-center py-12">
                <p className="body-text text-gray-500">No venues found</p>
              </div>
            ) : (
              <div className="grid gap-4 grid-cols-1 sm:grid-cols-2">
                {venues.map((venue) => (
                  <div
                    key={venue.id}
                    onMouseEnter={() => {
                      // Select venue for map highlighting on hover
                      setSelectedVenue(venue);
                    }}
                    className={`w-full ${
                      selectedVenue?.id === venue.id ? "" : ""
                    }`}
                  >
                    <VenueCard venue={venue} fullWidth />
                  </div>
                ))}
              </div>
            )}
          </div>
        </div>

        {/* Map Container - Always visible on tablet/desktop, toggleable on mobile */}
        <>
          {/* Mobile: Map overlay when toggled - shows full height */}
          {mapVisible && (
            <div className="sm:hidden fixed top-0 left-0 right-0 bottom-0 z-0 bg-white flex flex-col">
              {/* Map Container */}
              <div className="flex-1 relative" style={{ zIndex: 0 }}>
                {isMobileMapLoading ? (
                  <div className="flex items-center justify-center w-full h-full">
                    <div className="text-center">
                      <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900 mx-auto mb-4"></div>
                      <p className="body-text text-gray-500">Loading map...</p>
                    </div>
                  </div>
                ) : (
                  <div id="map-mobile" className="w-full h-full" style={{ zIndex: 0, position: 'relative' }} />
                )}
              </div>
            </div>
          )}
          {/* Tablet/Desktop: Side panel - always visible on tablet and desktop */}
          <div className="hidden sm:block w-full sm:w-1/2 p-4">
            <div className="sticky top-20 sm:top-24 h-[calc(100vh-4rem-2rem)] sm:h-[calc(100vh-5rem-30px-2rem)]">
              <div className="relative w-full h-full rounded-lg overflow-hidden bg-gray-100">
                {isMapLoading ? (
                  <div className="flex items-center justify-center w-full h-full absolute inset-0 z-10">
                    <div className="text-center">
                      <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900 mx-auto mb-4"></div>
                      <p className="body-text text-gray-500">Loading map...</p>
                    </div>
                  </div>
                ) : null}
                <div id="map-desktop" className="w-full h-full" />
              </div>
            </div>
          </div>
        </>
        {/* Hidden map container for initialization - always in DOM */}
        <div
          id="map"
          className="hidden"
          style={{ width: "400px", height: "400px" }}
        />
      </div>

      {/* Filter Modal */}
      <Dialog
        open={isFilterModalOpen}
        onOpenChange={setIsFilterModalOpen}
        showCloseButton={false}
        bottomSheet={true}
      >
        <div className="flex flex-col h-full max-w-2xl w-full min-h-0">
          {/* Modal Header - Fixed */}
          <div className="flex items-center justify-between p-6 pb-4 flex-shrink-0 border-b border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900">Filters</h2>
            <button
              onClick={() => setIsFilterModalOpen(false)}
              className="p-1 rounded-full hover:bg-gray-100 transition-colors"
            >
              <X className="w-5 h-5 text-gray-600" />
            </button>
          </div>

          {/* Scrollable Content */}
          <div className="flex-1 overflow-y-auto px-6 py-4 min-h-0">
            {/* Sort by */}
            <div className="mb-6">
              <h3 className="text-base font-semibold text-gray-900 mb-3">
                Sort by
              </h3>
              <div className="flex gap-2 sm:gap-3">
                <button
                  onClick={() => setSortBy("best-match")}
                  className={`flex-1 flex flex-col items-center gap-1 sm:gap-2 px-2 py-2 sm:px-4 sm:py-4 rounded-lg border transition-colors ${
                    sortBy === "best-match"
                      ? "border-purple-500 bg-purple-50"
                      : "border-gray-200 hover:bg-gray-50"
                  }`}
                >
                  <Heart
                    className={`w-4 h-4 sm:w-6 sm:h-6 ${
                      sortBy === "best-match"
                        ? "text-purple-700"
                        : "text-gray-700"
                    }`}
                  />
                  <span
                    className={`text-xs sm:text-base font-medium ${
                      sortBy === "best-match"
                        ? "text-purple-700"
                        : "text-gray-700"
                    }`}
                  >
                    Recommended
                  </span>
                </button>
                <button
                  onClick={() => setSortBy("nearest")}
                  className={`flex-1 flex flex-col items-center gap-1 sm:gap-2 px-2 py-2 sm:px-4 sm:py-4 rounded-lg border transition-colors ${
                    sortBy === "nearest"
                      ? "border-purple-500 bg-purple-50"
                      : "border-gray-200 hover:bg-gray-50"
                  }`}
                >
                  <MapPin
                    className={`w-4 h-4 sm:w-6 sm:h-6 ${
                      sortBy === "nearest" ? "text-purple-700" : "text-gray-700"
                    }`}
                  />
                  <span
                    className={`text-xs sm:text-base font-medium ${
                      sortBy === "nearest" ? "text-purple-700" : "text-gray-700"
                    }`}
                  >
                    Nearest
                  </span>
                </button>
                <button
                  onClick={() => setSortBy("top-rated")}
                  className={`flex-1 flex flex-col items-center gap-1 sm:gap-2 px-2 py-2 sm:px-4 sm:py-4 rounded-lg border transition-colors ${
                    sortBy === "top-rated"
                      ? "border-purple-500 bg-purple-50"
                      : "border-gray-200 hover:bg-gray-50"
                  }`}
                >
                  <Star
                    className={`w-4 h-4 sm:w-6 sm:h-6 ${
                      sortBy === "top-rated"
                        ? "text-purple-700"
                        : "text-gray-700"
                    }`}
                  />
                  <span
                    className={`text-xs sm:text-base font-medium ${
                      sortBy === "top-rated"
                        ? "text-purple-700"
                        : "text-gray-700"
                    }`}
                  >
                    Top rated
                  </span>
                </button>
              </div>
            </div>

            {/* Maximum price */}
            <div className="mb-6">
              <div className="flex items-center justify-between mb-3">
                <h3 className="text-base font-semibold text-gray-900">
                  Maximum price
                </h3>
                <span className="text-base font-semibold text-gray-900">
                  £{maxPrice}
                </span>
              </div>
              <input
                type="range"
                min="0"
                max="1000"
                step="10"
                value={maxPrice}
                onChange={(e) => setMaxPrice(parseInt(e.target.value))}
                className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer accent-purple-600"
              />
              <div className="flex items-center justify-between mt-1 text-xs text-gray-500">
                <span>£0</span>
                <span>£1000</span>
              </div>
            </div>

            {/* Rating */}
            <div className="mb-6">
              <h3 className="text-base font-semibold text-gray-900 mb-3">
                Rating
              </h3>
              <div className="space-y-2">
                {[4.5, 4.0, 3.5, 3.0].map((rating) => (
                  <button
                    key={rating}
                    onClick={() =>
                      setMinRating(minRating === rating ? null : rating)
                    }
                    className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg border transition-colors ${
                      minRating === rating
                        ? "border-purple-500 bg-purple-50 text-purple-700"
                        : "border-gray-200 hover:bg-gray-50 text-gray-700"
                    }`}
                  >
                    <div className="flex items-center gap-0.5">
                      {[1, 2, 3, 4, 5].map((star) => {
                        const isFilled = star <= rating;
                        const isHalfFilled =
                          star === Math.ceil(rating) &&
                          rating % 1 !== 0 &&
                          star > rating;
                        return (
                          <Star
                            key={star}
                            className={`w-5 h-5 ${
                              isFilled
                                ? "fill-yellow-400 text-yellow-400"
                                : isHalfFilled
                                ? "fill-yellow-400/50 text-yellow-400/50"
                                : "fill-none text-gray-300"
                            }`}
                          />
                        );
                      })}
                    </div>
                    <span className="font-medium">{rating}+</span>
                  </button>
                ))}
              </div>
            </div>
          </div>

          {/* Action Buttons - Fixed at Bottom */}
          <div className="flex gap-3 p-6 pt-4 border-t md:rounded-b-lg border-gray-200 bg-white flex-shrink-0">
            <Button
              variant="secondary"
              className="flex-1"
              onClick={() => {
                setSortBy("best-match");
                setMaxPrice(700);
                setMinRating(null);
                // Also apply the cleared filters immediately
                setAppliedSortBy("best-match");
                setAppliedMaxPrice(700);
                setAppliedMinRating(null);
              }}
            >
              Clear all
            </Button>
            <Button
              variant="primary"
              className="flex-1"
              onClick={() => {
                // Apply the filters
                setAppliedSortBy(sortBy);
                setAppliedMaxPrice(maxPrice);
                setAppliedMinRating(minRating);
                setIsFilterModalOpen(false);
              }}
            >
              Apply
            </Button>
          </div>
        </div>
      </Dialog>
    </div>
  );
}

// Extend Window interface for Google Maps
declare global {
  interface Window {
    google: typeof google;
  }
}
