"use client";

import { useState, useEffect, useRef } from "react";
import { MapPin, X } from "lucide-react";
import { googleApiKey } from "@/config/env";

interface LocationSearchProps {
  isOpen: boolean;
  onClose: () => void;
  onSelect: (location: { name: string; lat: number; lng: number } | null) => void;
  selectedLocation: { name: string; lat: number; lng: number } | null;
}

export function LocationSearch({
  isOpen,
  onClose,
  onSelect,
  selectedLocation,
}: LocationSearchProps) {
  const [searchQuery, setSearchQuery] = useState("");
  const [predictions, setPredictions] = useState<any[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const autocompleteServiceRef = useRef<google.maps.places.AutocompleteService | null>(null);

  useEffect(() => {
    if (isOpen && typeof window !== "undefined" && window.google) {
      autocompleteServiceRef.current = new window.google.maps.places.AutocompleteService();
    } else if (isOpen && typeof window !== "undefined") {
      // Load Google Maps script if not loaded
      const script = document.createElement("script");
      script.src = `https://maps.googleapis.com/maps/api/js?key=${googleApiKey}&libraries=places`;
      script.async = true;
      script.defer = true;
      script.onload = () => {
        autocompleteServiceRef.current = new window.google.maps.places.AutocompleteService();
      };
      document.head.appendChild(script);
    }
  }, [isOpen]);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        dropdownRef.current &&
        !dropdownRef.current.contains(event.target as Node)
      ) {
        onClose();
      }
    };

    if (isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
    }

    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, [isOpen, onClose]);

  useEffect(() => {
    if (!searchQuery || !autocompleteServiceRef.current) {
      setPredictions([]);
      return;
    }

    const timer = setTimeout(() => {
      setIsLoading(true);
      autocompleteServiceRef.current?.getPlacePredictions(
        {
          input: searchQuery,
          types: ["(cities)"],
        },
        (predictions, status) => {
          setIsLoading(false);
          if (status === google.maps.places.PlacesServiceStatus.OK && predictions) {
            setPredictions(predictions);
          } else {
            setPredictions([]);
          }
        }
      );
    }, 300);

    return () => clearTimeout(timer);
  }, [searchQuery]);

  const handleSelectPrediction = (prediction: google.maps.places.AutocompletePrediction) => {
    const geocoder = new google.maps.Geocoder();
    geocoder.geocode({ placeId: prediction.place_id }, (results, status) => {
      if (status === "OK" && results && results[0]) {
        const location = results[0].geometry.location;
        onSelect({
          name: prediction.description,
          lat: location.lat(),
          lng: location.lng(),
        });
        onClose();
      }
    });
  };

  if (!isOpen) return null;

  return (
    <div
      ref={dropdownRef}
      className="absolute top-full left-0 mt-2 bg-white rounded-xl shadow-lg border border-gray-200 z-50 max-h-96 overflow-hidden flex flex-col"
      style={{ width: "312px" }}
    >
      {/* Search Input */}
      <div className="p-4 border-b border-gray-200">
        <div className="relative">
          <MapPin className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
          <input
            type="text"
            placeholder="Search location..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-500"
          />
        </div>
      </div>

      {/* Predictions List */}
      <div className="overflow-y-auto flex-1">
        {isLoading ? (
          <div className="p-8 text-center text-gray-500">Searching...</div>
        ) : predictions.length === 0 && searchQuery ? (
          <div className="p-8 text-center text-gray-500">No locations found</div>
        ) : (
          <>
            {/* Current location option */}
            <button
              onClick={() => {
                if (navigator.geolocation) {
                  navigator.geolocation.getCurrentPosition(
                    (position) => {
                      onSelect({
                        name: "Current location",
                        lat: position.coords.latitude,
                        lng: position.coords.longitude,
                      });
                      onClose();
                    },
                    () => {
                      onSelect(null);
                      onClose();
                    }
                  );
                } else {
                  onSelect(null);
                  onClose();
                }
              }}
              className={`w-full px-4 py-3 text-left hover:bg-gray-50 flex items-center gap-3 ${
                selectedLocation?.name === "Current location" ? "bg-purple-50" : ""
              }`}
            >
              <MapPin className="w-5 h-5 text-gray-500" />
              <span className="body-text">Current location</span>
            </button>

            {/* Location predictions */}
            {predictions.map((prediction) => (
              <button
                key={prediction.place_id}
                onClick={() => handleSelectPrediction(prediction)}
                className="w-full px-4 py-3 text-left hover:bg-gray-50 flex items-center gap-3"
              >
                <MapPin className="w-5 h-5 text-gray-500" />
                <div className="flex flex-col">
                  <span className="body-text">{prediction.structured_formatting.main_text}</span>
                  <span className="body-text-sm text-gray-500">
                    {prediction.structured_formatting.secondary_text}
                  </span>
                </div>
              </button>
            ))}
          </>
        )}
      </div>
    </div>
  );
}

// Extend Window interface for Google Maps
declare global {
  interface Window {
    google: typeof google;
  }
}

