/**
 * Search Salons API - Client Side Only
 * Searches salons using the customer/search endpoint
 */

"use client";

import { clientFetcher } from "@/lib/api/fetcher-client";
import { z } from "zod";
import { HomeSalonSchema, type HomeSalon } from "@/lib/schemas/home";

// clientFetcher already extracts data.data, so we parse just the data object
const SearchSalonsDataSchema = z.object({
  slons: z.array(HomeSalonSchema),
  appointments: z.array(z.unknown()).optional(),
  currencyUnit: z.string(),
  distanceUnit: z.string(),
  highestPriceAcrossAllSalons: z.string(),
  lowestPriceAcrossAllSalons: z.string(),
});

export interface SearchSalonsRequest {
  targetLat: number;
  targetLng: number;
  rangeInKm?: number;
  day?: string;
  time?: string;
  price?: number;
  serviceType?: number;
}

export interface SearchSalonsResponse {
  slons: HomeSalon[];
  currencyUnit: string;
  distanceUnit: string;
  highestPriceAcrossAllSalons: string;
  lowestPriceAcrossAllSalons: string;
}

/**
 * Client-side: Search salons
 */
export async function searchSalonsClient(
  request: SearchSalonsRequest
): Promise<SearchSalonsResponse> {
  const data = await clientFetcher<unknown>("/customer/search", {
    requireAuth: false,
    method: "POST",
    body: JSON.stringify(request),
  });

  // clientFetcher already extracts data.data, so we parse the data object directly
  const parsed = SearchSalonsDataSchema.parse(data);
  return parsed;
}
