"use client";

import { useState } from "react";
import { Heart, Share2, Check } from "lucide-react";

interface SalonActionButtonsProps {
  salonName?: string | null;
  isLiked?: boolean;
  onLikeToggle?: () => void;
}

export function SalonActionButtons({
  salonName,
  isLiked = false,
  onLikeToggle,
}: SalonActionButtonsProps) {
  const [liked, setLiked] = useState(isLiked);
  const [shareCopied, setShareCopied] = useState(false);

  const handleLike = () => {
    setLiked(!liked);
    onLikeToggle?.();
  };

  const handleShare = async () => {
    const url = window.location.href;
    const title = salonName || "Salon";
    const text = `Check out ${title} on Fresha`;

    try {
      // Try Web Share API first (works on mobile and modern browsers)
      if (navigator.share) {
        await navigator.share({
          title,
          text,
          url,
        });
        return;
      }

      // Fallback: Copy to clipboard
      await navigator.clipboard.writeText(url);
      setShareCopied(true);
      setTimeout(() => setShareCopied(false), 2000);
    } catch (error) {
      // User cancelled share or error occurred
      // If Web Share API failed, try clipboard fallback
      if (typeof navigator.clipboard !== "undefined" && navigator.clipboard.writeText) {
        try {
          await navigator.clipboard.writeText(url);
          setShareCopied(true);
          setTimeout(() => setShareCopied(false), 2000);
        } catch (clipboardError) {
          console.error("Failed to copy to clipboard:", clipboardError);
        }
      }
    }
  };

  return (
    <div className="flex gap-2 w-full">
      <button
        onClick={handleShare}
        className="flex-1 h-10 rounded-lg border border-gray-300 flex items-center justify-center hover:bg-gray-50 transition-colors relative"
        aria-label="Share"
      >
        {shareCopied ? (
          <Check className="w-5 h-5 text-green-600" />
        ) : (
          <Share2 className="w-5 h-5 text-gray-900" />
        )}
      </button>
      <button
        onClick={handleLike}
        className="flex-1 h-10 rounded-lg border border-gray-300 flex items-center justify-center hover:bg-gray-50 transition-colors"
        aria-label={liked ? "Remove from favorites" : "Add to favorites"}
      >
        <Heart
          className={`w-5 h-5 ${
            liked ? "fill-red-500 text-red-500" : "text-gray-900"
          }`}
        />
      </button>
    </div>
  );
}
