/**
 * Transform Appointment from API to display format
 */

import { Appointment } from "@/lib/schemas/appointment";
import { API_BASE_URL } from "@/config/env";

export interface AppointmentDisplay {
  id: number;
  venueName: string;
  date: string; // Formatted date like "Fri, 14 Feb 2025"
  time: string; // Formatted time like "1:15 pm"
  fullDate: string; // Full date and time string
  cost: string; // Formatted cost like "Rs 170"
  items: number;
  image: string;
  status: string;
  address?: string;
  duration?: number; // Duration in minutes
  bookingReference?: string;
  services?: Array<{
    name: string;
    price: string;
    duration?: number;
    professionalName?: string;
  }>;
}

export function transformAppointmentToDisplay(
  appointment: Appointment,
  currencyUnit: string = "Rs"
): AppointmentDisplay {
  // Format date
  const appointmentDate = new Date(`${appointment.on}T${appointment.startTime}`);
  const formattedDate = appointmentDate.toLocaleDateString("en-US", {
    weekday: "short",
    day: "numeric",
    month: "short",
    year: "numeric",
  });

  // Format time
  const timeParts = appointment.startTime.split(":");
  const hours = parseInt(timeParts[0], 10);
  const minutes = timeParts[1];
  const ampm = hours >= 12 ? "pm" : "am";
  const displayHours = hours % 12 || 12;
  const formattedTime = `${displayHours}:${minutes} ${ampm}`;

  // Calculate duration
  const startTimeParts = appointment.startTime.split(":").map(Number);
  const endTimeParts = appointment.endTime.split(":").map(Number);
  const startMinutes = startTimeParts[0] * 60 + startTimeParts[1];
  const endMinutes = endTimeParts[0] * 60 + endTimeParts[1];
  const duration = endMinutes - startMinutes;

  // Format full date string
  const fullDate = `${formattedDate} at ${formattedTime}`;

  // Format cost with currency unit from API
  const cost = `${currencyUnit} ${parseFloat(appointment.total).toFixed(2)}`;

  // Build address string
  const address = appointment.salonDetail.addressDB
    ? `${appointment.salonDetail.addressDB.streetAddress}, ${appointment.salonDetail.addressDB.city}`
    : undefined;

  // Handle image URL - convert relative paths to absolute URLs
  // Use placeholder URL that can be detected for showing placeholder component
  let imageUrl = "/placeholder-venue";
  if (appointment.salonDetail.coverImage) {
    if (
      appointment.salonDetail.coverImage.startsWith("http://") ||
      appointment.salonDetail.coverImage.startsWith("https://")
    ) {
      imageUrl = appointment.salonDetail.coverImage;
    } else if (appointment.salonDetail.coverImage.startsWith("/")) {
      imageUrl = `${API_BASE_URL}${appointment.salonDetail.coverImage}`;
    } else {
      // Relative path like "Public/Gallery/..." - prepend base URL with leading slash
      imageUrl = `${API_BASE_URL}/${appointment.salonDetail.coverImage}`;
    }
  }

  return {
    id: appointment.id,
    venueName: appointment.salonDetail.salonName,
    date: formattedDate,
    time: formattedTime,
    fullDate,
    cost,
    items: appointment.serviceCount,
    image: imageUrl,
    status: appointment.status,
    address,
    duration,
    bookingReference: appointment.id.toString(),
  };
}

