/**
 * React Query hook for favorites
 */

import { useQuery } from "@tanstack/react-query";
import { getFavoritesClient } from "../api/getFavorites-client";
import type { FavoriteVenue, GetFavoritesRequest } from "@/lib/schemas/favorites";

interface UseFavoritesOptions {
  location?: {
    lat: number;
    lng: number;
    rangeInKm?: number;
  };
}

export function useFavorites(options?: UseFavoritesOptions) {
  return useQuery<{ slons: FavoriteVenue[] }>({
    queryKey: [
      "favorites",
      options?.location?.lat,
      options?.location?.lng,
      options?.location?.rangeInKm || 15,
    ],
    queryFn: async () => {
      if (!options?.location) {
        throw new Error("Location is required");
      }

      const request: GetFavoritesRequest = {
        targetLat: options.location.lat,
        targetLng: options.location.lng,
        rangeInKm: options.location.rangeInKm || 15,
      };

      return await getFavoritesClient(request);
    },
    enabled: !!options?.location?.lat && !!options?.location?.lng,
    staleTime: 5 * 60 * 1000, // 5 minutes
    retry: 1,
  });
}

