"use client";

import { useState, useEffect } from "react";
import { Dialog } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Checkbox } from "@/components/ui/checkbox";
import { RadioGroup, RadioItem } from "@/components/ui/radio-group";
import { getAccountDeletionReasonsClient } from "../api/getAccountDeletionReasons-client";
import { deleteCustomerClient } from "../api/deleteCustomer-client";
import type { DeletionReason, UserData } from "@/lib/schemas/accountDeletion";
import { CheckCircle2, Loader2 } from "lucide-react";
import { getUserData } from "@/lib/auth";

interface DeleteAccountModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSuccess?: () => void;
}

type Step = 1 | 2 | 3 | 4 | 5;

export function DeleteAccountModal({
  open,
  onOpenChange,
  onSuccess,
}: DeleteAccountModalProps) {
  const [step, setStep] = useState<Step>(1);
  const [reasons, setReasons] = useState<DeletionReason[]>([]);
  const [userData, setUserData] = useState<UserData | null>(null);
  const [selectedReasonId, setSelectedReasonId] = useState<number | null>(null);
  const [checkbox1, setCheckbox1] = useState(false);
  const [checkbox2, setCheckbox2] = useState(false);
  const [checkbox1Error, setCheckbox1Error] = useState("");
  const [checkbox2Error, setCheckbox2Error] = useState("");
  const [email, setEmail] = useState("");
  const [emailError, setEmailError] = useState("");
  const [apiError, setApiError] = useState<string | null>(null);
  const [isLoading, setIsLoading] = useState(false);
  const [isDeleting, setIsDeleting] = useState(false);
  const [storedEmail, setStoredEmail] = useState<string | null>(null);

  // Fetch reasons and user data when modal opens
  useEffect(() => {
    if (open && step === 1) {
      fetchReasons();
      // Get stored email from localStorage
      const userData = getUserData();
      if (userData?.email) {
        setStoredEmail(userData.email);
      }
    }
  }, [open, step]);

  // Auto-advance from step 4 to step 5
  useEffect(() => {
    if (step === 4) {
      const timer = setTimeout(() => {
        setStep(5);
      }, 1200);
      return () => clearTimeout(timer);
    }
  }, [step]);

  const fetchReasons = async () => {
    setIsLoading(true);
    try {
      const data = await getAccountDeletionReasonsClient();
      setReasons(data.reasons);
      setUserData(data.userData);
    } catch (error) {
      console.error("Failed to fetch deletion reasons:", error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleStep1Continue = () => {
    if (selectedReasonId) {
      setStep(2);
    }
  };

  const handleStep2Continue = () => {
    let hasError = false;
    if (!checkbox1) {
      setCheckbox1Error("Please confirm");
      hasError = true;
    } else {
      setCheckbox1Error("");
    }
    if (!checkbox2) {
      setCheckbox2Error("Please confirm");
      hasError = true;
    } else {
      setCheckbox2Error("");
    }

    if (!hasError) {
      setStep(3);
    }
  };

  const handleStep3Delete = async () => {
    // Validate email
    if (!email.trim()) {
      setEmailError("Email is required");
      return;
    }

    // Check email matches stored email from localStorage
    if (
      storedEmail &&
      email.trim().toLowerCase() !== storedEmail.toLowerCase()
    ) {
      setEmailError("Email does not match your account email");
      return;
    }

    if (!selectedReasonId) {
      return;
    }

    setIsDeleting(true);
    setApiError(null);
    setEmailError("");
    try {
      await deleteCustomerClient({ reasonId: selectedReasonId });
      setStep(4);
    } catch (error: any) {
      console.error("Failed to delete account:", error);
      // Check if it's an ApiError with a specific message
      if (error?.message) {
        setApiError(error.message);
      } else {
        setApiError("Failed to delete account. Please try again.");
      }
    } finally {
      setIsDeleting(false);
    }
  };

  const handleBack = () => {
    if (step === 2) {
      setStep(1);
      setCheckbox1Error("");
      setCheckbox2Error("");
    } else if (step === 3) {
      setStep(2);
      setEmailError("");
      setEmail("");
      setApiError(null);
    }
  };

  const handleClose = () => {
    if (step === 5) {
      onOpenChange(false);
      if (onSuccess) {
        onSuccess();
      }
    } else {
      onOpenChange(false);
    }
    // Reset state
    setTimeout(() => {
      setStep(1);
      setSelectedReasonId(null);
      setCheckbox1(false);
      setCheckbox2(false);
      setCheckbox1Error("");
      setCheckbox2Error("");
      setEmail("");
      setEmailError("");
      setApiError(null);
    }, 300);
  };

  const renderStep1 = () => (
    <div className="space-y-6">
      <div className="space-y-2">
        <h2 className="heading-4 text-gray-900">
          We'll be sorry to see you go
        </h2>
        <p className="body-text-sm text-gray-600">
          Can we change your mind? Drop us a line at hello@fresha.com
        </p>
      </div>

      <div className="space-y-4">
        <div>
          <h3 className="body-text-sm font-medium text-gray-900 mb-1">
            Why do you wish to leave Fresha?
          </h3>
          <p className="caption text-gray-500 mb-4">
            Please let us know the reason for deleting your account:
          </p>
        </div>

        {isLoading ? (
          <div className="flex items-center justify-center py-8">
            <Loader2 className="w-6 h-6 text-gray-400 animate-spin" />
          </div>
        ) : (
          <RadioGroup
            value={selectedReasonId?.toString()}
            onValueChange={(value) => setSelectedReasonId(parseInt(value))}
          >
            {reasons.map((reason) => (
              <RadioItem
                key={reason.id}
                value={reason.id.toString()}
                label={reason.reason}
              />
            ))}
          </RadioGroup>
        )}
      </div>

      <div className="pt-4 border-t border-gray-200 mt-auto">
        <Button
          onClick={handleStep1Continue}
          disabled={!selectedReasonId || isLoading}
          className="w-full"
        >
          Continue →
        </Button>
      </div>
    </div>
  );

  const renderStep2 = () => (
    <div className="space-y-6">
      <div className="space-y-2">
        <h2 className="heading-4 text-gray-900">Delete your Fresha account</h2>
        <p className="body-text-sm text-gray-600">
          This action will delete your account and you won't be able to retrieve
          it.
        </p>
        <p className="body-text-sm text-gray-600">
          Please confirm you understand by ticking the below statements:
        </p>
      </div>

      <div className="space-y-4">
        <Checkbox
          id="checkbox1"
          checked={checkbox1}
          onChange={(e) => {
            setCheckbox1(e.target.checked);
            if (e.target.checked) {
              setCheckbox1Error("");
            }
          }}
          label="I know I won't be able to access my appointments, gift cards or memberships."
          error={checkbox1Error}
          touched={!!checkbox1Error}
        />

        <Checkbox
          id="checkbox2"
          checked={checkbox2}
          onChange={(e) => {
            setCheckbox2(e.target.checked);
            if (e.target.checked) {
              setCheckbox2Error("");
            }
          }}
          label="I know I will no longer be able to book appointments via Fresha's Marketplace."
          error={checkbox2Error}
          touched={!!checkbox2Error}
        />
      </div>

      <div className="pt-4 border-t border-gray-200 mt-auto">
        <Button
          onClick={handleStep2Continue}
          disabled={!checkbox1 || !checkbox2}
          className="w-full"
        >
          Continue →
        </Button>
      </div>
    </div>
  );

  const renderStep3 = () => {
    // Check if email matches stored email
    const emailMatches =
      storedEmail && email.trim().toLowerCase() === storedEmail.toLowerCase();
    const isFormValid = email.trim() && emailMatches;

    return (
      <div className="space-y-6">
        <div className="space-y-2">
          <h2 className="heading-4 text-gray-900">
            Confirm you are happy to proceed
          </h2>
          <p className="body-text-sm text-gray-600">
            Please note this is permanent and can't be undone.
          </p>
          <p className="body-text-sm text-gray-600">
            To confirm deleting your account, please enter your email below:
          </p>
        </div>

        <div className="space-y-4">
          {apiError && (
            <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
              <p className="text-sm text-red-600">{apiError}</p>
            </div>
          )}
          <Input
            label="Email"
            type="email"
            value={email}
            onChange={(e) => {
              setEmail(e.target.value);
              if (e.target.value.trim()) {
                setEmailError("");
              }
              // Clear API error when user starts typing
              if (apiError) {
                setApiError(null);
              }
            }}
            error={emailError}
            touched={!!emailError}
            placeholder="Enter your email"
          />
        </div>

        <div className="pt-4 border-t border-gray-200 mt-auto">
          <Button
            onClick={handleStep3Delete}
            disabled={isDeleting || !isFormValid}
            className="w-full bg-red-600 text-white hover:bg-red-700"
          >
            {isDeleting ? (
              <>
                <Loader2 className="w-4 h-4 animate-spin" />
                Deleting account...
              </>
            ) : (
              "Delete account"
            )}
          </Button>
        </div>
      </div>
    );
  };

  const renderStep4 = () => (
    <div className="space-y-6">
      <div className="flex flex-col items-center justify-center py-12 space-y-4">
        <CheckCircle2 className="w-16 h-16 text-green-500" />
        <p className="heading-5 text-gray-900">Your account has been deleted</p>
      </div>
    </div>
  );

  const renderStep5 = () => (
    <div className="space-y-6">
      <div className="flex flex-col items-center justify-center py-12 space-y-4">
        <CheckCircle2 className="w-16 h-16 text-green-500" />
        <h2 className="heading-4 text-gray-900">
          Your account has been deleted
        </h2>
        <p className="body-text-sm text-gray-600 text-center">
          We're sorry to see you go.
        </p>
        <p className="body-text-sm text-gray-600 text-center">
          If you need any further help, please send a message to
          hello@fresha.com
        </p>
      </div>

      <div className="pt-4 border-t border-gray-200 mt-auto">
        <Button onClick={handleClose} className="w-full">
          Done
        </Button>
      </div>
    </div>
  );

  return (
    <Dialog
      open={open}
      onOpenChange={handleClose}
      showBackButton={step === 2 || step === 3}
      onBack={handleBack}
      showCloseButton={step !== 4 && step !== 5}
    >
      <div className="p-6 flex flex-col min-h-[400px]">
        <h3 className="heading-5 text-gray-900 mb-6">Delete my account</h3>
        <div className="flex-1 flex flex-col">
          {step === 1 && renderStep1()}
          {step === 2 && renderStep2()}
          {step === 3 && renderStep3()}
          {step === 4 && renderStep4()}
          {step === 5 && renderStep5()}
        </div>
      </div>
    </Dialog>
  );
}
