"use client";

import { useState, useEffect } from "react";
import { Search, MapPin, Calendar, Clock, X } from "lucide-react";
import { useRouter } from "next/navigation";
import { Container } from "@/components/layout/Container";
import { ServiceTypesDropdown } from "@/features/search/components/ServiceTypesDropdown";
import { LocationSearch } from "@/features/search/components/LocationSearch";
import { DatePicker } from "@/features/search/components/DatePicker";
import { TimePicker } from "@/features/search/components/TimePicker";
import { ServiceType } from "@/features/search/api/getServiceTypes-client";

export function HeroSection() {
  const router = useRouter();
  const [serviceDropdownOpen, setServiceDropdownOpen] = useState(false);
  const [locationDropdownOpen, setLocationDropdownOpen] = useState(false);
  const [datePickerOpen, setDatePickerOpen] = useState(false);
  const [timePickerOpen, setTimePickerOpen] = useState(false);
  const [selectedService, setSelectedService] = useState<ServiceType | null>(
    null
  );
  const [selectedLocation, setSelectedLocation] = useState<{
    name: string;
    lat: number;
    lng: number;
  } | null>(null);
  const [selectedDate, setSelectedDate] = useState<{
    type: "any" | "today" | "tomorrow" | "date";
    date?: Date;
  } | null>(null);
  const [selectedTime, setSelectedTime] = useState<{
    period?: "any" | "morning" | "afternoon" | "evening";
    from?: string;
    to?: string;
  } | null>(null);

  // Get default location or use geolocation
  useEffect(() => {
    if (!selectedLocation && navigator.geolocation) {
      navigator.geolocation.getCurrentPosition(
        (position) => {
          setSelectedLocation({
            name: "Current location",
            lat: position.coords.latitude,
            lng: position.coords.longitude,
          });
        },
        () => {
          // Fallback to default location
          setSelectedLocation({
            name: "Current location",
            lat: 31.462576,
            lng: 74.247932,
          });
        }
      );
    } else if (!selectedLocation) {
      setSelectedLocation({
        name: "Current location",
        lat: 31.462576,
        lng: 74.247932,
      });
    }
  }, []);

  const handleSearch = () => {
    const params = new URLSearchParams();
    if (selectedLocation) {
      params.set("lat", selectedLocation.lat.toString());
      params.set("lng", selectedLocation.lng.toString());
    } else {
      // Use default location if not selected
      params.set("lat", "31.462576");
      params.set("lng", "74.247932");
    }
    if (selectedService) {
      params.set("serviceType", selectedService.id.toString());
    }
    if (selectedDate && selectedDate.type !== "any" && selectedDate.date) {
      const dayNames = [
        "Sunday",
        "Monday",
        "Tuesday",
        "Wednesday",
        "Thursday",
        "Friday",
        "Saturday",
      ];
      params.set("day", dayNames[selectedDate.date.getDay()]);
    }
    if (selectedTime && selectedTime.from && selectedTime.to) {
      // Convert time to 24-hour format for API
      const from24 = convertTo24Hour(selectedTime.from);
      params.set("time", from24);
    }
    router.push(`/search?${params.toString()}`);
  };

  const convertTo24Hour = (time12: string): string => {
    const [time, period] = time12.split(" ");
    const [hours, minutes] = time.split(":");
    let hour24 = parseInt(hours);
    if (period === "pm" && hour24 !== 12) hour24 += 12;
    if (period === "am" && hour24 === 12) hour24 = 0;
    return `${hour24.toString().padStart(2, "0")}:${minutes}:00`;
  };

  const getDateDisplayText = () => {
    if (!selectedDate) return "Any date";
    if (selectedDate.type === "any") return "Any date";
    if (selectedDate.type === "today") return "Today";
    if (selectedDate.type === "tomorrow") return "Tomorrow";
    if (selectedDate.type === "date" && selectedDate.date) {
      return selectedDate.date.toLocaleDateString("en-US", {
        weekday: "short",
        month: "short",
        day: "numeric",
      });
    }
    return "Any date";
  };

  const getTimeDisplayText = () => {
    if (!selectedTime || selectedTime.period === "any") return "Any time";
    if (selectedTime.from && selectedTime.to) {
      return `${selectedTime.from} - ${selectedTime.to}`;
    }
    return "Any time";
  };

  return (
    <section className="relative min-h-[50vh] sm:min-h-[60vh] md:min-h-[70vh] lg:min-h-[80vh] flex items-center justify-center pt-12 sm:pt-16 md:pt-20 pb-6 sm:pb-8 md:pb-12 px-2 sm:px-4 bg-gradient-to-b from-purple-50 via-pink-50 to-white">
      <Container className="relative z-10">
        {/* Top Copy */}
        <div className="text-center mb-6 sm:mb-8 md:mb-10">
          <h1
            className="text-3xl sm:text-4xl md:text-5xl lg:text-6xl font-bold mb-3 sm:mb-4 px-2"
            style={{ color: "var(--text-primary)" }}
          >
            Book local selfcare services
          </h1>
          <p
            className="text-sm sm:text-base md:text-lg max-w-4xl mx-auto px-2 sm:px-4"
            style={{ color: "var(--text-secondary)" }}
          >
            Discover top-rated salons, barbers, medspas, wellness studios and
            beauty experts trusted by millions worldwide.
          </p>
        </div>

        {/* Search Bar Block */}
        <div className="max-w-4xl mx-auto mb-4 sm:mb-6 px-2">
          <div className="relative flex flex-col lg:flex-row items-stretch lg:items-center bg-white rounded-2xl lg:rounded-full shadow-md lg:p-2">
            {/* Service Input */}
            <div className="relative flex-1">
              <div className="relative">
                <button
                  type="button"
                  onClick={() => {
                    setServiceDropdownOpen(!serviceDropdownOpen);
                    setLocationDropdownOpen(false);
                    setDatePickerOpen(false);
                  }}
                  className="w-full pl-12 pr-4 py-4 lg:py-3.5 text-left hover:bg-gray-50 lg:rounded-l-full transition-colors text-sm sm:text-base text-gray-900 focus:outline-none focus:ring-0 border-0 flex items-center relative"
                >
                  <Search className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-500 pointer-events-none" />
                  <span className="flex-1 truncate">
                    {selectedService
                      ? selectedService.typeName
                      : "All treatments and venues"}
                  </span>
                </button>
              </div>
              {serviceDropdownOpen && (
                <div className="absolute left-0 right-0 lg:left-0 lg:right-0 top-full mt-2 z-50">
                  <ServiceTypesDropdown
                    isOpen={serviceDropdownOpen}
                    onClose={() => setServiceDropdownOpen(false)}
                    onSelect={(service) => {
                      setSelectedService(service);
                      setServiceDropdownOpen(false);
                    }}
                    selectedService={selectedService}
                    location={selectedLocation}
                  />
                </div>
              )}
            </div>

            {/* Border Divider - Mobile horizontal, Desktop vertical */}
            <div className="lg:hidden h-px bg-gray-200" />
            <div className="hidden lg:block w-px h-8 bg-gray-300 self-center" />

            {/* Location Field */}
            <div className="relative flex-1">
              <button
                onClick={() => {
                  setLocationDropdownOpen(!locationDropdownOpen);
                  setServiceDropdownOpen(false);
                  setDatePickerOpen(false);
                }}
                className="w-full flex items-center gap-3 px-4 py-4 lg:py-3.5 text-left hover:bg-gray-50 transition-colors border-0 text-sm sm:text-base"
              >
                <MapPin className="w-5 h-5 text-gray-500 flex-shrink-0" />
                <span className="flex-1 text-gray-900 truncate">
                  {selectedLocation
                    ? selectedLocation.name
                    : "Current location"}
                </span>
              </button>
              {locationDropdownOpen && (
                <div className="absolute left-0 right-0 lg:left-0 lg:right-0 top-full mt-2 z-50">
                  <LocationSearch
                    isOpen={locationDropdownOpen}
                    onClose={() => setLocationDropdownOpen(false)}
                    onSelect={setSelectedLocation}
                    selectedLocation={selectedLocation}
                  />
                </div>
              )}
            </div>

            {/* Border Divider - Mobile horizontal, Desktop vertical */}
            <div className="lg:hidden h-px bg-gray-200" />
            <div className="hidden lg:block w-px h-8 bg-gray-300 self-center" />

            {/* Date & Time Field - Combined */}
            <div className="relative flex-1">
              <button
                onClick={() => {
                  setDatePickerOpen(!datePickerOpen);
                  setServiceDropdownOpen(false);
                  setLocationDropdownOpen(false);
                }}
                className="w-full flex items-center gap-3 px-4 py-4 lg:py-3.5 text-left hover:bg-gray-50 transition-colors border-0 text-sm sm:text-base"
              >
                <Calendar className="w-5 h-5 text-gray-500 flex-shrink-0" />
                <span className="flex-1 text-gray-900 truncate">
                  {selectedDate && selectedDate.type !== "any"
                    ? `${getDateDisplayText()}${
                        selectedTime && selectedTime.period !== "any"
                          ? ` · ${getTimeDisplayText()}`
                          : " · Any time"
                      }`
                    : "Today · Any time"}
                </span>
              </button>
              {datePickerOpen && (
                <div className="absolute left-0 right-0 lg:left-auto lg:right-0 lg:w-[600px] top-full mt-2 z-50">
                  <div className="bg-white rounded-2xl shadow-xl p-6">
                    {/* Date Picker */}
                    <DatePicker
                      isOpen={datePickerOpen}
                      onClose={() => setDatePickerOpen(false)}
                      onSelect={setSelectedDate}
                      selectedDate={selectedDate}
                    />
                    
                    {/* Time Picker */}
                    <div className="mt-6 pt-6 border-t border-gray-200">
                      <TimePicker
                        isOpen={true}
                        onClose={() => {}}
                        onSelect={setSelectedTime}
                        selectedTime={selectedTime}
                      />
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Border Divider - Only on mobile */}
            <div className="lg:hidden h-px bg-gray-200" />

            {/* Search Button */}
            <button
              onClick={handleSearch}
              className="w-full lg:w-auto px-8 py-4 lg:py-3.5 bg-black text-white rounded-b-2xl lg:rounded-full hover:bg-gray-800 transition-colors flex items-center gap-2 justify-center whitespace-nowrap text-sm sm:text-base font-medium"
            >
              <span className="lg:hidden">Search Fresha</span>
              <span className="hidden lg:inline">Search</span>
            </button>
          </div>

          {/* Appointments Count */}
          <p
            className="text-sm sm:text-base text-center mt-4 sm:mt-6"
            style={{ color: "var(--text-secondary)" }}
          >
            <span className="font-bold">298,347</span> appointments booked today
          </p>
        </div>

        {/* Get the App CTA */}
        <div className="text-center mt-6 sm:mt-8">
          <button className="px-4 py-2 sm:px-6 sm:py-3 bg-white border border-gray-300 rounded-xl hover:bg-gray-50 transition-colors inline-flex items-center gap-2 text-sm sm:text-base">
            <span>Get the app</span>
            <div className="flex gap-0.5 items-center">
              <div className="w-3 h-3 sm:w-3.5 sm:h-3.5 border border-gray-400 rounded-sm"></div>
              <div className="w-3 h-3 sm:w-3.5 sm:h-3.5 border border-gray-400 rounded-sm -ml-1"></div>
            </div>
          </button>
        </div>
      </Container>
    </section>
  );
}
