/**
 * Get Home Venues API - Server Side Only
 * Fetches nearby salons/venues for the home page based on location
 */

import "server-only";
import {
  GetHomeVenuesRequestSchema,
  GetHomeVenuesResponseSchema,
  type GetHomeVenuesRequest,
  type GetHomeVenuesResponse,
} from "@/lib/schemas/home";

/**
 * Server-side: Get home venues
 * Use in Server Components, Server Actions, Route Handlers
 * Dynamically imports server fetcher to avoid bundling in client code
 */
export async function getHomeVenuesServer(
  request: GetHomeVenuesRequest
): Promise<GetHomeVenuesResponse> {
  // Validate request
  const validatedRequest = GetHomeVenuesRequestSchema.parse(request);

  // Dynamically import server fetcher to avoid bundling in client
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const data = await serverFetcher<unknown>("/customer/home", {
    requireAuth: false, // Public endpoint
    method: "POST",
    body: JSON.stringify(validatedRequest),
  });

  // Parse with better error handling
  try {
    return GetHomeVenuesResponseSchema.parse(data);
  } catch (error) {
    // Log the raw data for debugging
    console.error("Zod validation failed. Raw data:", JSON.stringify(data, null, 2));
    throw error;
  }
}






