/**
 * OTP Verification API
 * Verifies OTP and completes account setup
 */

import { clientFetcher } from "@/lib/api/fetcher-client";
import { API_ENDPOINTS } from "@/config/env";
import { getDeviceToken } from "@/lib/deviceToken";

export interface VerifyOtpRequest {
  otpId: number;
  userId: number;
  OTP: number;
  dvToken: string;
}

export interface UserData {
  accessToken: string;
  userId: string;
  image: string;
  firstName: string;
  lastName: string;
  email: string;
  phoneNum: string;
  stripeCustomerId: string | null;
  joinOn: string;
}

export interface VerifyOtpResponse {
  accessToken: string;
  userId: string;
  image: string;
  firstName: string;
  lastName: string;
  email: string;
  phoneNum: string;
  stripeCustomerId: string | null;
  joinOn: string;
}

/**
 * Verify OTP and complete account setup
 * Returns user data with access token
 * @param otpId - OTP ID from signup response
 * @param userId - User ID from signup response
 * @param otp - OTP code (will be converted to number)
 */
export async function verifyOtp(
  otpId: number,
  userId: number,
  otp: number | string
): Promise<VerifyOtpResponse> {
  const dvToken = getDeviceToken();

  // Convert OTP to number (API expects number, not string)
  const otpNumber = typeof otp === "string" ? parseInt(otp, 10) : otp;

  if (isNaN(otpNumber)) {
    throw new Error("Invalid OTP format");
  }

  const response = await clientFetcher<VerifyOtpResponse>(API_ENDPOINTS.verifyOtp, {
    requireAuth: false,
    method: "POST",
    body: JSON.stringify({
      otpId,
      userId,
      OTP: otpNumber,
      dvToken,
    }),
  });

  return response;
}

