import { SelectHTMLAttributes, forwardRef } from "react";
import { cn } from "@/lib/classNames";
import { ChevronDown, AlertCircle } from "lucide-react";

interface SelectProps extends SelectHTMLAttributes<HTMLSelectElement> {
  label?: string;
  error?: string;
  touched?: boolean;
}

export const Select = forwardRef<HTMLSelectElement, SelectProps>(
  ({ label, error, touched, className = "", children, ...props }, ref) => {
    const showError = touched && error;

    return (
      <div className="w-full">
        {label && (
          <label
            className={cn(
              "label block mb-2",
              showError && "text-error-500"
            )}
          >
            {label}
          </label>
        )}
        <div className="relative">
          <select
            ref={ref}
            className={cn(
              "w-full px-4 py-3 rounded-xl border appearance-none transition-colors",
              "body-text pr-10",
              showError
                ? "border-error-500 focus:border-error-500 focus:ring-error-500 bg-error-50"
                : "border-gray-300 focus:border-gray-900 focus:ring-gray-900 bg-white",
              "focus:outline-none focus:ring-2 focus:ring-offset-0",
              "cursor-pointer",
              className
            )}
            aria-invalid={showError ? true : undefined}
            aria-describedby={showError ? `${props.id || "select"}-error` : undefined}
            {...props}
          >
            {children}
          </select>
          <ChevronDown className={cn(
            "absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 pointer-events-none",
            showError ? "text-error-500" : "text-gray-400"
          )} />
        </div>
        {showError && (
          <p
            id={`${props.id || "select"}-error`}
            className="caption text-error-500 mt-1 flex items-center gap-1"
            role="alert"
          >
            <AlertCircle className="w-3 h-3" />
            {error}
          </p>
        )}
      </div>
    );
  }
);

Select.displayName = "Select";
