import { createContext, useContext, ReactNode } from "react";
import { cn } from "@/lib/classNames";

interface RadioGroupContextValue {
  value?: string;
  onValueChange?: (value: string) => void;
}

const RadioGroupContext = createContext<RadioGroupContextValue>({});

interface RadioGroupProps {
  value?: string;
  onValueChange?: (value: string) => void;
  children: ReactNode;
  className?: string;
}

export function RadioGroup({
  value,
  onValueChange,
  children,
  className = "",
}: RadioGroupProps) {
  return (
    <RadioGroupContext.Provider value={{ value, onValueChange }}>
      <div className={cn("space-y-3", className)} role="radiogroup">
        {children}
      </div>
    </RadioGroupContext.Provider>
  );
}

interface RadioItemProps {
  value: string;
  label: string;
  className?: string;
}

export function RadioItem({ value, label, className = "" }: RadioItemProps) {
  const { value: selectedValue, onValueChange } = useContext(RadioGroupContext);
  const isSelected = selectedValue === value;

  return (
    <label
      className={cn(
        "flex items-start gap-3 p-4 rounded-lg border cursor-pointer transition-colors",
        isSelected
          ? "border-purple-500 bg-purple-50"
          : "border-gray-200 hover:border-gray-300",
        className
      )}
    >
      <input
        type="radio"
        value={value}
        checked={isSelected}
        onChange={() => onValueChange?.(value)}
        className="mt-1 w-5 h-5 text-purple-600 focus:ring-2 focus:ring-purple-500 focus:ring-offset-0 cursor-pointer flex-shrink-0"
      />
      <span className="body-text-sm text-gray-900 flex-1">{label}</span>
    </label>
  );
}
