"use client";

import {
  Search,
  MapPin,
  Clock,
  ChevronDown,
  Calendar,
  ArrowLeft,
  SlidersHorizontal,
} from "lucide-react";
import Image from "next/image";
import Link from "next/link";
import { routes } from "@/config/routes";
import { useState, useEffect, useRef } from "react";
import { isAuthenticated, getUserData, clearAuth } from "@/lib/auth";
import { API_BASE_URL } from "@/config/env";
import { useRouter, usePathname } from "next/navigation";
import { ServiceTypesDropdown } from "@/features/search/components/ServiceTypesDropdown";
import { LocationSearch } from "@/features/search/components/LocationSearch";
import { DatePicker } from "@/features/search/components/DatePicker";
import { TimePicker } from "@/features/search/components/TimePicker";
import {
  ServiceType,
  getServiceTypesClient,
} from "@/features/search/api/getServiceTypes-client";
import { useSearchParams } from "next/navigation";

export function ProfileHeader() {
  const router = useRouter();
  const pathname = usePathname();
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [userData, setUserData] = useState<any>(null);
  const [showUserMenu, setShowUserMenu] = useState(false);
  const [imageError, setImageError] = useState(false);

  // Search states
  const [serviceDropdownOpen, setServiceDropdownOpen] = useState(false);
  const [locationDropdownOpen, setLocationDropdownOpen] = useState(false);
  const [datePickerOpen, setDatePickerOpen] = useState(false);
  const [selectedService, setSelectedService] = useState<ServiceType | null>(
    null
  );
  const [selectedLocation, setSelectedLocation] = useState<{
    name: string;
    lat: number;
    lng: number;
  } | null>(null);
  const [selectedDate, setSelectedDate] = useState<{
    type: "any" | "today" | "tomorrow" | "date";
    date?: Date;
  } | null>(null);
  const [selectedTime, setSelectedTime] = useState<{
    period?: "any" | "morning" | "afternoon" | "evening";
    from?: string;
    to?: string;
  } | null>(null);
  const [isSearchBarActive, setIsSearchBarActive] = useState(false);
  const [isScrolled, setIsScrolled] = useState(false);
  const [isMapVisible, setIsMapVisible] = useState(false);
  const [isMobileSearchModalOpen, setIsMobileSearchModalOpen] = useState(false);
  const [isMobileSearchModalClosing, setIsMobileSearchModalClosing] =
    useState(false);
  const [venueCount, setVenueCount] = useState<string>("");

  useEffect(() => {
    const checkAuth = () => {
      const authenticated = isAuthenticated();
      setIsLoggedIn(authenticated);
      if (authenticated) {
        const data = getUserData();
        setUserData(data);
        setImageError(false);
      } else {
        setUserData(null);
        setImageError(false);
      }
    };

    checkAuth();
    // Check auth state on mount, when storage changes (across tabs), and on custom auth state change event (same tab)
    window.addEventListener("storage", checkAuth);
    window.addEventListener("authStateChanged", checkAuth);
    return () => {
      window.removeEventListener("storage", checkAuth);
      window.removeEventListener("authStateChanged", checkAuth);
    };
  }, [pathname]);

  const searchParams = useSearchParams();

  // Initialize selected service from URL params
  useEffect(() => {
    const serviceTypeId = searchParams?.get("serviceType");
    if (serviceTypeId) {
      const fetchService = async () => {
        try {
          const services = await getServiceTypesClient();
          const service = services.find(
            (s) => s.id.toString() === serviceTypeId
          );
          if (service) {
            setSelectedService(service);
          }
        } catch (error) {
          console.error("Failed to fetch service types:", error);
        }
      };
      fetchService();
    }
  }, [searchParams]);

  // Get default location or use geolocation
  useEffect(() => {
    if (!selectedLocation && navigator.geolocation) {
      navigator.geolocation.getCurrentPosition(
        (position) => {
          setSelectedLocation({
            name: "Current location",
            lat: position.coords.latitude,
            lng: position.coords.longitude,
          });
        },
        () => {
          // Fallback to default location
          setSelectedLocation({
            name: "Current location",
            lat: 31.462576,
            lng: 74.247932,
          });
        }
      );
    } else if (!selectedLocation) {
      setSelectedLocation({
        name: "Current location",
        lat: 31.462576,
        lng: 74.247932,
      });
    }
  }, []);

  // Close menu when clicking outside
  useEffect(() => {
    if (!showUserMenu) return;

    const handleClickOutside = (event: MouseEvent) => {
      const target = event.target as HTMLElement;
      if (!target.closest(".user-menu-container")) {
        setShowUserMenu(false);
      }
    };

    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, [showUserMenu]);

  // Listen for venue count updates from SearchResultsPage
  useEffect(() => {
    const handleVenueCountUpdate = (event: CustomEvent<string>) => {
      setVenueCount(event.detail);
    };
    window.addEventListener(
      "venueCountUpdate",
      handleVenueCountUpdate as EventListener
    );
    return () =>
      window.removeEventListener(
        "venueCountUpdate",
        handleVenueCountUpdate as EventListener
      );
  }, []);

  // Scroll detection for mobile header with smooth transitions
  useEffect(() => {
    let ticking = false;
    const handleScroll = () => {
      if (!ticking) {
        window.requestAnimationFrame(() => {
          const scrollY = window.scrollY;
          setIsScrolled(scrollY > 30);
          ticking = false;
        });
        ticking = true;
      }
    };

    window.addEventListener("scroll", handleScroll, { passive: true });
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  // Listen for map visibility changes
  useEffect(() => {
    const handleMapToggle = () => {
      setIsMapVisible((prev) => !prev);
    };
    window.addEventListener("toggleMap", handleMapToggle);
    return () => {
      window.removeEventListener("toggleMap", handleMapToggle);
    };
  }, []);

  const getFirstLetter = () => {
    if (userData?.firstName) {
      return userData.firstName.charAt(0).toUpperCase();
    }
    if (userData?.email) {
      return userData.email.charAt(0).toUpperCase();
    }
    return "U";
  };

  const hasProfileImage = () => {
    return userData?.image && userData.image.trim() !== "" && !imageError;
  };

  const getProfileImageUrl = () => {
    if (!userData?.image) return "";
    const imageUrl = userData.image.trim();
    if (imageUrl.startsWith("http://") || imageUrl.startsWith("https://")) {
      return imageUrl;
    }
    // Use API_BASE_URL for relative paths
    const baseUrl = API_BASE_URL.endsWith("/")
      ? API_BASE_URL.slice(0, -1)
      : API_BASE_URL;
    return imageUrl.startsWith("/") ? `${baseUrl}${imageUrl}` : imageUrl;
  };

  const handleSearch = () => {
    const params = new URLSearchParams();
    if (selectedLocation) {
      params.set("lat", selectedLocation.lat.toString());
      params.set("lng", selectedLocation.lng.toString());
    } else {
      params.set("lat", "31.462576");
      params.set("lng", "74.247932");
    }
    if (selectedService) {
      params.set("serviceType", selectedService.id.toString());
    }
    if (selectedDate && selectedDate.type !== "any" && selectedDate.date) {
      const dayNames = [
        "Sunday",
        "Monday",
        "Tuesday",
        "Wednesday",
        "Thursday",
        "Friday",
        "Saturday",
      ];
      params.set("day", dayNames[selectedDate.date.getDay()]);
    }
    if (selectedTime && selectedTime.from && selectedTime.to) {
      const from24 = convertTo24Hour(selectedTime.from);
      params.set("time", from24);
    }
    router.push(`/search?${params.toString()}`);
  };

  const convertTo24Hour = (time12: string): string => {
    const [time, period] = time12.split(" ");
    const [hours, minutes] = time.split(":");
    let hour24 = parseInt(hours);
    if (period === "pm" && hour24 !== 12) hour24 += 12;
    if (period === "am" && hour24 === 12) hour24 = 0;
    return `${hour24.toString().padStart(2, "0")}:${minutes}:00`;
  };

  const getDateDisplayText = () => {
    if (!selectedDate) return "Any time";
    if (selectedDate.type === "any") return "Any time";
    if (selectedDate.type === "today") return "Today";
    if (selectedDate.type === "tomorrow") return "Tomorrow";
    if (selectedDate.type === "date" && selectedDate.date) {
      return selectedDate.date.toLocaleDateString("en-US", {
        weekday: "short",
        month: "short",
        day: "numeric",
      });
    }
    return "Any time";
  };

  const getTimeDisplayText = () => {
    if (!selectedTime || selectedTime.period === "any") return "";
    if (selectedTime.from && selectedTime.to) {
      return ` · ${selectedTime.from} - ${selectedTime.to}`;
    }
    return "";
  };

  return (
    <>
      {/* Dark overlay when search or dropdown is active (excluding mobile modal) */}
      {(serviceDropdownOpen || locationDropdownOpen || datePickerOpen) &&
        !isMobileSearchModalOpen && (
          <div
            className="fixed inset-0 z-[45] transition-opacity duration-300"
            style={{ backgroundColor: "rgba(0, 0, 0, 0.4)" }}
            onClick={() => {
              setServiceDropdownOpen(false);
              setLocationDropdownOpen(false);
              setDatePickerOpen(false);
            }}
          />
        )}
      <header
        className={`lg:fixed lg:top-0 lg:left-0 lg:right-0 sticky top-0 z-[60] transition-all duration-300 ${
          isSearchBarActive ? "h-24" : "h-auto"
        } ${
          serviceDropdownOpen || locationDropdownOpen || datePickerOpen
            ? ""
            : "overflow-hidden"
        }`}
      >
        {/* Mobile Header - Floating design with fully rounded corners */}
        <div
          className={`sm:hidden transition-all duration-500 ease-in-out ${
            !isMapVisible ? "bg-white" : ""
          }`}
        >
          <div
            className={`px-4 transition-all duration-500 ease-in-out pt-2 pb-2 ${
              !isMapVisible ? "bg-white" : ""
            }`}
          >
            <div
              className={`rounded-full border border-gray-200 overflow-hidden transition-all duration-500 ease-in-out cursor-pointer ${
                isScrolled ? "py-1.5" : ""
              } ${isMapVisible ? "bg-white" : "bg-white"}`}
              onClick={() => {
                setIsMobileSearchModalClosing(false);
                setIsMobileSearchModalOpen(true);
              }}
            >
              {/* Top row: Back arrow + Service name + Date/Location + Map icon */}
              <div
                className={`flex items-center justify-between transition-all duration-500 ease-in-out ${
                  isScrolled ? "px-3 py-1.5" : "px-3 py-2"
                }`}
              >
                <div className="flex items-center gap-3 flex-1 min-w-0">
                  <button
                    onClick={(e) => {
                      e.stopPropagation();
                      router.back();
                    }}
                    className="p-1 hover:bg-gray-100 rounded-full transition-colors flex-shrink-0"
                  >
                    <ArrowLeft className="w-5 h-5 text-gray-700" />
                  </button>
                  <div
                    className={`flex-1 min-w-0 transition-all duration-500 ease-in-out ${
                      isScrolled ? "flex items-center gap-1.5" : "flex flex-col"
                    }`}
                  >
                    {/* Service name - shared element, truncates when scrolled */}
                    <h1
                      className={`text-sm font-semibold text-gray-900 transition-all duration-500 ease-in-out ${
                        isScrolled ? "truncate min-w-0" : "flex-shrink-0"
                      }`}
                    >
                      {selectedService
                        ? selectedService.typeName
                        : "All treatments"}
                    </h1>
                    {/* Date/location - transitions from below to inline */}
                    <div
                      className={`flex items-center gap-1.5 text-sm text-gray-500 transition-all duration-500 ease-in-out min-w-0 ${
                        isScrolled ? "opacity-100" : "opacity-100 mt-1"
                      }`}
                    >
                      <span className="text-xs text-gray-500 flex-shrink-0">
                        •
                      </span>
                      <span className="truncate min-w-0">
                        {selectedDate && selectedDate.type !== "any"
                          ? getDateDisplayText()
                          : "Any date"}
                      </span>
                      <span className="text-xs text-gray-500 flex-shrink-0">
                        •
                      </span>
                      <span className="truncate min-w-0">
                        {selectedLocation
                          ? selectedLocation.name
                          : "Current location"}
                      </span>
                    </div>
                  </div>
                </div>
                {/* Map icon button - circular with border, smaller on scroll */}
                <button
                  onClick={(e) => {
                    e.stopPropagation();
                    // Toggle map visibility - this will be handled by SearchResultsPage
                    const event = new CustomEvent("toggleMap");
                    window.dispatchEvent(event);
                  }}
                  className={`flex items-center justify-center bg-white border border-gray-300 rounded-full hover:bg-gray-50 transition-all duration-500 ease-in-out flex-shrink-0 ${
                    isScrolled ? "w-8 h-8" : "w-10 h-10"
                  }`}
                >
                  <MapPin
                    className={`text-gray-700 transition-all duration-500 ease-in-out ${
                      isScrolled ? "w-4 h-4" : "w-5 h-5"
                    }`}
                  />
                </button>
              </div>
            </div>
          </div>
          {/* Filter Buttons Row */}
          <div className={`px-4 pb-2 ${!isMapVisible ? "bg-white" : ""}`}>
            <div className="flex items-center justify-between gap-2">
              {/* Venue Count - Left Side */}
              {venueCount && (
                <p className="text-sm text-gray-600 font-medium">
                  {venueCount}
                </p>
              )}
              {/* Filters Button - Right Side */}
              <button
                onClick={() => {
                  // Dispatch custom event to open filter modal
                  window.dispatchEvent(new CustomEvent("openFilterModal"));
                }}
                className={`px-3 py-2 border border-gray-300 rounded-full hover:bg-gray-50 flex items-center gap-2 text-sm font-medium transition-colors whitespace-nowrap flex-shrink-0 ${
                  isMapVisible ? "bg-white" : ""
                }`}
              >
                <SlidersHorizontal className="w-4 h-4" />
                <span>Filters</span>
              </button>
            </div>
          </div>
        </div>

        {/* Desktop Header */}
        <div className="hidden sm:flex items-center justify-between h-20 px-4 sm:px-6 lg:px-8 relative bg-white">
          {/* Logo */}
          <Link
            href={routes.home}
            className="heading-6 lowercase text-gray-900"
          >
            fresha
          </Link>

          {/* Search Bar */}
          <div className="flex-1 max-w-2xl mx-8 relative">
            <div
              className={`flex items-center border border-gray-300 rounded-full transition-all duration-300 ${
                serviceDropdownOpen || locationDropdownOpen || datePickerOpen
                  ? "relative bg-white shadow-lg z-[60] scale-[1.01] origin-center"
                  : isSearchBarActive
                  ? "relative bg-white z-[60] scale-[1.01] origin-center"
                  : "relative bg-white"
              } ${
                serviceDropdownOpen || locationDropdownOpen || datePickerOpen
                  ? ""
                  : "overflow-hidden"
              }`}
              onClick={() => setIsSearchBarActive(true)}
            >
              {/* All treatments */}
              <div className="relative flex-1">
                <button
                  type="button"
                  onClick={(e) => {
                    e.stopPropagation();
                    setIsSearchBarActive(true);
                    setServiceDropdownOpen(!serviceDropdownOpen);
                    setLocationDropdownOpen(false);
                    setDatePickerOpen(false);
                  }}
                  className="w-full flex flex-col items-start px-4 py-3 border-r border-gray-300 hover:bg-gray-50 transition-colors"
                >
                  <div className="flex items-center gap-2 w-full">
                    <Search className="w-4 h-4 text-gray-500 flex-shrink-0" />
                    <span className="body-text-sm text-gray-700 truncate">
                      {selectedService
                        ? selectedService.typeName
                        : "All treatments"}
                    </span>
                  </div>
                </button>
                {serviceDropdownOpen && (
                  <div className="absolute left-0 top-full mt-2 z-[70]">
                    <ServiceTypesDropdown
                      isOpen={serviceDropdownOpen}
                      onClose={() => {
                        setServiceDropdownOpen(false);
                      }}
                      onSelect={(service) => {
                        setSelectedService(service);
                        setServiceDropdownOpen(false);
                        handleSearch();
                      }}
                      selectedService={selectedService}
                      location={selectedLocation}
                    />
                  </div>
                )}
              </div>

              {/* Current location */}
              <div className="relative flex-1">
                <button
                  onClick={(e) => {
                    e.stopPropagation();
                    setIsSearchBarActive(true);
                    setLocationDropdownOpen(!locationDropdownOpen);
                    setServiceDropdownOpen(false);
                    setDatePickerOpen(false);
                  }}
                  className="w-full flex flex-col items-start px-4 py-3 border-r border-gray-300 hover:bg-gray-50 transition-colors"
                >
                  <div className="flex items-center gap-2 w-full">
                    <MapPin className="w-4 h-4 text-gray-500 flex-shrink-0" />
                    <span className="body-text-sm text-gray-700 truncate">
                      {selectedLocation ? selectedLocation.name : "Map area"}
                    </span>
                  </div>
                </button>
                {locationDropdownOpen && (
                  <div className="absolute left-0 right-0 top-full mt-2 z-[70]">
                    <LocationSearch
                      isOpen={locationDropdownOpen}
                      onClose={() => {
                        setLocationDropdownOpen(false);
                      }}
                      onSelect={(location) => {
                        setSelectedLocation(location);
                        setLocationDropdownOpen(false);
                        handleSearch();
                      }}
                      selectedLocation={selectedLocation}
                    />
                  </div>
                )}
              </div>

              {/* Any time */}
              <div className="relative flex-1">
                <button
                  onClick={(e) => {
                    e.stopPropagation();
                    setIsSearchBarActive(true);
                    setDatePickerOpen(!datePickerOpen);
                    setServiceDropdownOpen(false);
                    setLocationDropdownOpen(false);
                  }}
                  className={`w-full flex flex-col items-start px-4 py-3 hover:bg-gray-50 transition-colors ${
                    isSearchBarActive ? "" : "border-r border-gray-300"
                  }`}
                >
                  <div className="flex items-center gap-2 w-full">
                    <Clock className="w-4 h-4 text-gray-500 flex-shrink-0" />
                    <span className="body-text-sm text-gray-700 truncate">
                      {getDateDisplayText()}
                      {getTimeDisplayText()}
                    </span>
                  </div>
                </button>
                {datePickerOpen && (
                  <div className="absolute right-0 top-full mt-2 z-[70] w-[600px]">
                    <div className="bg-white rounded-2xl shadow-xl border border-gray-200 z-50 p-6">
                      <DatePicker
                        isOpen={datePickerOpen}
                        onClose={() => {
                          setDatePickerOpen(false);
                          handleSearch();
                        }}
                        onSelect={(date) => {
                          setSelectedDate(date);
                          handleSearch();
                        }}
                        selectedDate={selectedDate}
                      />
                      <div className="mt-6 pt-6 border-t border-gray-200">
                        <TimePicker
                          isOpen={true}
                          onClose={() => {}}
                          onSelect={(time) => {
                            setSelectedTime(time);
                            handleSearch();
                          }}
                          selectedTime={selectedTime}
                        />
                      </div>
                    </div>
                  </div>
                )}
              </div>

              {/* Search Button - Animated */}
              <div
                className={`overflow-hidden transition-all duration-300 ${
                  isSearchBarActive
                    ? "max-w-[120px] opacity-100"
                    : "max-w-0 opacity-0"
                }`}
              >
                <button
                  onClick={(e) => {
                    e.stopPropagation();
                    handleSearch();
                    setIsSearchBarActive(false);
                  }}
                  className="px-6 py-3 bg-black text-white rounded-full hover:bg-gray-800 transition-colors font-medium text-sm whitespace-nowrap flex-shrink-0"
                >
                  Search
                </button>
              </div>
            </div>
          </div>

          {/* User Profile */}
          {isLoggedIn && (
            <div className="relative user-menu-container">
              <button
                onClick={() => setShowUserMenu(!showUserMenu)}
                className="flex items-center gap-2 px-3 py-2 bg-white border border-gray-300 rounded-full hover:bg-gray-50 transition-colors"
              >
                {hasProfileImage() ? (
                  <div className="w-8 h-8 rounded-full overflow-hidden relative flex-shrink-0">
                    <Image
                      src={getProfileImageUrl()}
                      alt={`${userData.firstName || "User"} profile`}
                      fill
                      className="object-cover"
                      onError={() => setImageError(true)}
                    />
                  </div>
                ) : (
                  <div className="w-8 h-8 rounded-full bg-teal-500 flex items-center justify-center text-white font-semibold text-sm flex-shrink-0">
                    {getFirstLetter()}
                  </div>
                )}
                <ChevronDown className="w-4 h-4 text-gray-700 flex-shrink-0" />
              </button>
              {showUserMenu && (
                <div className="absolute right-0 top-full mt-2 w-48 bg-white rounded-xl border border-gray-200 shadow-lg py-2 z-[200]">
                  <Link
                    href={routes.profile}
                    onClick={() => setShowUserMenu(false)}
                    className="block px-4 py-2 body-text-sm hover:bg-gray-50 transition-colors"
                  >
                    Profile
                  </Link>
                  <Link
                    href={routes.profileAppointments}
                    onClick={() => setShowUserMenu(false)}
                    className="block px-4 py-2 body-text-sm hover:bg-gray-50 transition-colors"
                  >
                    Appointments
                  </Link>
                  <Link
                    href={routes.profileSettings}
                    onClick={() => setShowUserMenu(false)}
                    className="block px-4 py-2 body-text-sm hover:bg-gray-50 transition-colors"
                  >
                    Settings
                  </Link>
                  <div className="border-t border-gray-200 my-2"></div>
                  <button
                    onClick={() => {
                      clearAuth();
                      setIsLoggedIn(false);
                      setUserData(null);
                      setShowUserMenu(false);
                      router.push(routes.home);
                    }}
                    className="w-full text-left px-4 py-2 body-text-sm hover:bg-gray-50 transition-colors text-error-500"
                  >
                    Log out
                  </button>
                </div>
              )}
            </div>
          )}
        </div>
      </header>

      {/* Mobile Search Modal - Bottom Sheet Popup */}
      {(isMobileSearchModalOpen || isMobileSearchModalClosing) && (
        <div className="sm:hidden fixed inset-0 z-[100] flex items-end pointer-events-none">
          {/* Backdrop - Black Overlay */}
          <div
            className={`absolute inset-0 pointer-events-auto transition-opacity duration-300 ${
              isMobileSearchModalClosing ? "opacity-0" : "opacity-100"
            }`}
            style={{ backgroundColor: "rgba(0, 0, 0, 0.4)" }}
            onClick={() => {
              setIsMobileSearchModalClosing(true);
              setTimeout(() => {
                setIsMobileSearchModalOpen(false);
                setIsMobileSearchModalClosing(false);
              }, 300);
            }}
          />

          {/* Modal Content */}
          <div
            className={`relative w-full bg-white rounded-t-3xl shadow-2xl pointer-events-auto max-h-[90vh] flex flex-col transition-transform duration-300 ease-out ${
              isMobileSearchModalClosing ? "translate-y-full" : "translate-y-0"
            }`}
            onClick={(e) => e.stopPropagation()}
            style={{
              animation: isMobileSearchModalClosing
                ? "none"
                : "slide-up 0.3s ease-out",
            }}
          >
            {/* Drag Handle */}
            <div className="flex justify-center pt-3 pb-2 flex-shrink-0">
              <div className="w-12 h-1.5 bg-gray-300 rounded-full" />
            </div>

            {/* Modal Header */}
            <div className="px-6 pt-2 pb-4 border-b border-gray-200 flex-shrink-0">
              <h2 className="text-2xl font-bold text-gray-900">Search</h2>
            </div>

            {/* Modal Content */}
            <div className="px-6 py-6 space-y-4 overflow-y-auto flex-1">
              {/* Service Input */}
              <div className="relative">
                <button
                  type="button"
                  onClick={(e) => {
                    e.stopPropagation();
                    setServiceDropdownOpen(!serviceDropdownOpen);
                    setLocationDropdownOpen(false);
                    setDatePickerOpen(false);
                  }}
                  className="w-full flex items-center gap-3 px-4 py-4 border border-gray-300 rounded-lg bg-white hover:bg-gray-50 transition-colors text-left"
                >
                  <Search className="w-5 h-5 text-gray-500 flex-shrink-0" />
                  <span className="flex-1 text-base text-gray-700">
                    {selectedService
                      ? selectedService.typeName
                      : "Eyebrows & eyelashes"}
                  </span>
                </button>
                {serviceDropdownOpen && (
                  <div className="absolute left-0 right-0 top-full mt-2 z-[70]">
                    <ServiceTypesDropdown
                      isOpen={serviceDropdownOpen}
                      onClose={() => setServiceDropdownOpen(false)}
                      onSelect={(service) => {
                        setSelectedService(service);
                        setServiceDropdownOpen(false);
                      }}
                      selectedService={selectedService}
                      location={selectedLocation}
                    />
                  </div>
                )}
              </div>

              {/* Location Input */}
              <div className="relative">
                <button
                  type="button"
                  onClick={(e) => {
                    e.stopPropagation();
                    setLocationDropdownOpen(!locationDropdownOpen);
                    setServiceDropdownOpen(false);
                    setDatePickerOpen(false);
                  }}
                  className="w-full flex items-center gap-3 px-4 py-4 border border-gray-300 rounded-lg bg-white hover:bg-gray-50 transition-colors text-left"
                >
                  <MapPin className="w-5 h-5 text-gray-500 flex-shrink-0" />
                  <span className="flex-1 text-base text-gray-700">
                    {selectedLocation
                      ? selectedLocation.name
                      : "Current location"}
                  </span>
                </button>
                {locationDropdownOpen && (
                  <div className="absolute left-0 right-0 top-full mt-2 z-[70]">
                    <LocationSearch
                      isOpen={locationDropdownOpen}
                      onClose={() => setLocationDropdownOpen(false)}
                      onSelect={(location) => {
                        setSelectedLocation(location);
                        setLocationDropdownOpen(false);
                      }}
                      selectedLocation={selectedLocation}
                    />
                  </div>
                )}
              </div>

              {/* Date/Time Input */}
              <div className="relative">
                <button
                  type="button"
                  onClick={(e) => {
                    e.stopPropagation();
                    setDatePickerOpen(!datePickerOpen);
                    setServiceDropdownOpen(false);
                    setLocationDropdownOpen(false);
                  }}
                  className="w-full flex items-center gap-3 px-4 py-4 border border-gray-300 rounded-lg bg-white hover:bg-gray-50 transition-colors text-left"
                >
                  <Calendar className="w-5 h-5 text-gray-500 flex-shrink-0" />
                  <span className="flex-1 text-base text-gray-700">
                    {selectedDate && selectedDate.type !== "any"
                      ? getDateDisplayText()
                      : "Any time"}
                    {selectedDate &&
                      selectedDate.type !== "any" &&
                      getTimeDisplayText()}
                  </span>
                </button>
                {datePickerOpen && (
                  <div className="absolute left-0 right-0 top-full mt-2 z-[70] max-h-[400px] overflow-y-auto">
                    <div className="bg-white rounded-xl shadow-xl border border-gray-200 z-50 p-4">
                      <DatePicker
                        isOpen={datePickerOpen}
                        onClose={() => {
                          setDatePickerOpen(false);
                        }}
                        onSelect={(date) => {
                          setSelectedDate(date);
                        }}
                        selectedDate={selectedDate}
                      />
                      <div className="mt-4 pt-4 border-t border-gray-200">
                        <TimePicker
                          isOpen={true}
                          onClose={() => {}}
                          onSelect={(time) => {
                            setSelectedTime(time);
                          }}
                          selectedTime={selectedTime}
                        />
                      </div>
                    </div>
                  </div>
                )}
              </div>
            </div>

            {/* Search Button */}
            <div className="px-6 pb-6 pt-4 border-t border-gray-200 flex-shrink-0 safe-area-inset-bottom">
              <button
                onClick={() => {
                  handleSearch();
                  setIsMobileSearchModalClosing(true);
                  setTimeout(() => {
                    setIsMobileSearchModalOpen(false);
                    setIsMobileSearchModalClosing(false);
                  }, 300);
                }}
                className="w-full py-4 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors font-medium text-base"
              >
                Search
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  );
}
