"use client";

import { Menu, ChevronDown, User, Calendar, Heart, Settings, X } from "lucide-react";
import Image from "next/image";
import Link from "next/link";
import { routes } from "@/config/routes";
import { Container } from "@/components/layout/Container";
import { useState, useEffect } from "react";
import { isAuthenticated, getUserData, clearAuth } from "@/lib/auth";
import { useRouter, usePathname } from "next/navigation";

export function Header() {
  const router = useRouter();
  const pathname = usePathname();
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [userData, setUserData] = useState<any>(null);
  const [showUserMenu, setShowUserMenu] = useState(false);
  const [showMobileSidebar, setShowMobileSidebar] = useState(false);
  const [imageError, setImageError] = useState(false);

  useEffect(() => {
    const checkAuth = () => {
      const authenticated = isAuthenticated();
      setIsLoggedIn(authenticated);
      if (authenticated) {
        const data = getUserData();
        setUserData(data);
        // Reset image error when user data changes
        setImageError(false);
      } else {
        setUserData(null);
        setImageError(false);
      }
    };

    checkAuth();
    // Check auth state on mount and when storage changes
    window.addEventListener("storage", checkAuth);
    return () => window.removeEventListener("storage", checkAuth);
  }, []);

  // Close menu when clicking outside
  useEffect(() => {
    if (!showUserMenu) return;

    const handleClickOutside = (event: MouseEvent) => {
      const target = event.target as HTMLElement;
      if (!target.closest(".user-menu-container")) {
        setShowUserMenu(false);
      }
    };

    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, [showUserMenu]);

  // Prevent body scroll when mobile sidebar is open
  useEffect(() => {
    if (showMobileSidebar) {
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "";
    }
    return () => {
      document.body.style.overflow = "";
    };
  }, [showMobileSidebar]);

  const handleLogout = () => {
    clearAuth();
    setIsLoggedIn(false);
    setUserData(null);
    setShowUserMenu(false);
    setShowMobileSidebar(false);
    router.push(routes.home);
  };

  const handleNavClick = (href: string) => {
    setShowMobileSidebar(false);
    router.push(href);
  };

  const getFirstLetter = () => {
    // Return first letter of first name, or email if no first name
    if (userData?.firstName) {
      return userData.firstName.charAt(0).toUpperCase();
    }
    if (userData?.email) {
      return userData.email.charAt(0).toUpperCase();
    }
    return "U";
  };

  const hasProfileImage = () => {
    return userData?.image && userData.image.trim() !== "" && !imageError;
  };

  const getProfileImageUrl = () => {
    if (!userData?.image) return "";
    const imageUrl = userData.image.trim();
    // If it's already a full URL, return it
    if (imageUrl.startsWith("http://") || imageUrl.startsWith("https://")) {
      return imageUrl;
    }
    // Otherwise, prepend API base URL if it's a relative path
    return imageUrl.startsWith("/")
      ? `https://stage.trimworldwide.com${imageUrl}`
      : imageUrl;
  };

  return (
    <header
      className="sticky top-0 z-50 bg-white border-b border-gray-100"
      style={{
        background:
          "linear-gradient(to right, rgba(255, 255, 255, 1), rgba(243, 232, 255, 0.2))",
      }}
    >
      <Container>
        <div className="flex items-center justify-between h-20">
          {/* Logo */}
          <Link href={routes.home} className="heading-6 lowercase">
            fresha
          </Link>

          {/* Navigation */}
          <nav className="hidden md:flex items-center gap-6">
            {!isLoggedIn ? (
              <>
                <Link
                  href={routes.login}
                  className="nav-link hover:text-gray-700 transition-colors"
                >
                  Log in
                </Link>
                <Link
                  href={routes.business}
                  className="px-4 py-2 bg-white border border-gray-300 rounded-full button-text hover:bg-gray-50 transition-colors"
                >
                  List your business
                </Link>
                <button className="px-4 py-2 bg-white border border-gray-300 rounded-full button-text hover:bg-gray-50 transition-colors flex items-center gap-2">
                  <span>Menu</span>
                  <Menu className="w-4 h-4" />
                </button>
              </>
            ) : (
              <>
                <Link
                  href={routes.business}
                  className="px-4 py-2 bg-white border border-gray-300 rounded-full button-text hover:bg-gray-50 transition-colors"
                >
                  List your business
                </Link>
                {/* User Profile Dropdown */}
                <div className="relative user-menu-container">
                  <button
                    onClick={() => setShowUserMenu(!showUserMenu)}
                    className="px-3 py-2 bg-white border border-gray-300 rounded-full button-text hover:bg-gray-50 transition-colors flex items-center gap-2"
                  >
                    {hasProfileImage() ? (
                      <div className="w-8 h-8 rounded-full overflow-hidden flex-shrink-0 relative">
                        <Image
                          src={getProfileImageUrl()}
                          alt={`${userData.firstName || "User"} profile`}
                          fill
                          className="object-cover"
                          onError={() => setImageError(true)}
                        />
                      </div>
                    ) : (
                      <div className="w-8 h-8 rounded-full bg-teal-500 flex items-center justify-center text-white font-semibold text-sm flex-shrink-0">
                        {getFirstLetter()}
                      </div>
                    )}
                    <ChevronDown className="w-4 h-4 text-gray-700" />
                  </button>
                  {showUserMenu && (
                    <div className="absolute right-0 mt-2 w-48 bg-white rounded-xl border border-gray-200 shadow-lg py-2 z-50">
                      <Link
                        href={routes.profile}
                        onClick={() => setShowUserMenu(false)}
                        className="block px-4 py-2 body-text-sm hover:bg-gray-50 transition-colors"
                      >
                        Profile
                      </Link>
                      <Link
                        href={routes.profileAppointments}
                        onClick={() => setShowUserMenu(false)}
                        className="block px-4 py-2 body-text-sm hover:bg-gray-50 transition-colors"
                      >
                        Appointments
                      </Link>
                      <Link
                        href={routes.profileSettings}
                        onClick={() => setShowUserMenu(false)}
                        className="block px-4 py-2 body-text-sm hover:bg-gray-50 transition-colors"
                      >
                        Settings
                      </Link>
                      <div className="border-t border-gray-200 my-2"></div>
                      <button
                        onClick={handleLogout}
                        className="w-full text-left px-4 py-2 body-text-sm hover:bg-gray-50 transition-colors text-error-500"
                      >
                        Log out
                      </button>
                    </div>
                  )}
                </div>
              </>
            )}
          </nav>

          {/* Mobile Profile Icon */}
          <div className="md:hidden relative user-menu-container">
            {!isLoggedIn ? (
              <Link
                href={routes.login}
                className="p-2 text-gray-700 hover:text-gray-900 transition-colors"
              >
                <div className="w-8 h-8 rounded-full bg-gray-200 flex items-center justify-center">
                  <span className="text-sm font-semibold text-gray-600">U</span>
                </div>
              </Link>
            ) : (
              <button
                onClick={() => setShowMobileSidebar(true)}
                className="p-2 text-gray-700 hover:text-gray-900 transition-colors"
              >
                {hasProfileImage() ? (
                  <div className="w-8 h-8 rounded-full overflow-hidden flex-shrink-0 relative">
                    <Image
                      src={getProfileImageUrl()}
                      alt={`${userData.firstName || "User"} profile`}
                      fill
                      className="object-cover"
                      onError={() => setImageError(true)}
                    />
                  </div>
                ) : (
                  <div className="w-8 h-8 rounded-full bg-teal-500 flex items-center justify-center text-white font-semibold text-sm flex-shrink-0">
                    {getFirstLetter()}
                  </div>
                )}
              </button>
            )}
          </div>
        </div>
      </Container>

      {/* Mobile/Tablet Sidebar */}
      {showMobileSidebar && isLoggedIn && (
        <>
          {/* Overlay */}
          <div
            className="fixed inset-0 bg-black/50 z-[60] lg:hidden"
            onClick={() => setShowMobileSidebar(false)}
          />
          {/* Sidebar */}
          <aside className="fixed top-0 right-0 h-full w-80 max-w-[85vw] bg-white z-[70] lg:hidden shadow-2xl animate-slide-in-right">
            <div className="h-full flex flex-col">
              {/* Header */}
              <div className="flex items-center justify-between p-6 border-b border-gray-200">
                <div className="flex items-center gap-3">
                  {hasProfileImage() ? (
                    <div className="w-12 h-12 rounded-full overflow-hidden flex-shrink-0 relative">
                      <Image
                        src={getProfileImageUrl()}
                        alt={`${userData.firstName || "User"} profile`}
                        fill
                        className="object-cover"
                        onError={() => setImageError(true)}
                      />
                    </div>
                  ) : (
                    <div className="w-12 h-12 rounded-full bg-teal-500 flex items-center justify-center text-white font-semibold text-lg flex-shrink-0">
                      {getFirstLetter()}
                    </div>
                  )}
                  <div>
                    <h2 className="text-lg font-bold text-gray-900">
                      {userData?.firstName && userData?.lastName
                        ? `${userData.firstName} ${userData.lastName}`
                        : userData?.email || "User"}
                    </h2>
                    <p className="text-sm text-gray-500">Personal profile</p>
                  </div>
                </div>
                <button
                  onClick={() => setShowMobileSidebar(false)}
                  className="p-2 hover:bg-gray-100 rounded-full transition-colors"
                >
                  <X className="w-5 h-5 text-gray-600" />
                </button>
              </div>

              {/* Navigation Items */}
              <nav className="flex-1 overflow-y-auto p-4">
                <div className="space-y-1">
                  <button
                    onClick={() => handleNavClick(routes.profile)}
                    className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${
                      pathname === routes.profile
                        ? "bg-purple-50 text-purple-700 font-medium"
                        : "text-gray-700 hover:bg-gray-50"
                    }`}
                  >
                    <User
                      className={`w-5 h-5 ${
                        pathname === routes.profile ? "text-purple-700" : "text-gray-500"
                      }`}
                    />
                    <span className="text-sm">Profile</span>
                  </button>
                  <button
                    onClick={() => handleNavClick(routes.profileAppointments)}
                    className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${
                      pathname === routes.profileAppointments
                        ? "bg-purple-50 text-purple-700 font-medium"
                        : "text-gray-700 hover:bg-gray-50"
                    }`}
                  >
                    <Calendar
                      className={`w-5 h-5 ${
                        pathname === routes.profileAppointments ? "text-purple-700" : "text-gray-500"
                      }`}
                    />
                    <span className="text-sm">Appointments</span>
                  </button>
                  <button
                    onClick={() => handleNavClick(routes.profileFavorites)}
                    className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${
                      pathname === routes.profileFavorites
                        ? "bg-purple-50 text-purple-700 font-medium"
                        : "text-gray-700 hover:bg-gray-50"
                    }`}
                  >
                    <Heart
                      className={`w-5 h-5 ${
                        pathname === routes.profileFavorites ? "text-purple-700" : "text-gray-500"
                      }`}
                    />
                    <span className="text-sm">Favorites</span>
                  </button>
                  <button
                    onClick={() => handleNavClick(routes.profileSettings)}
                    className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${
                      pathname === routes.profileSettings
                        ? "bg-purple-50 text-purple-700 font-medium"
                        : "text-gray-700 hover:bg-gray-50"
                    }`}
                  >
                    <Settings
                      className={`w-5 h-5 ${
                        pathname === routes.profileSettings ? "text-purple-700" : "text-gray-500"
                      }`}
                    />
                    <span className="text-sm">Settings</span>
                  </button>
                </div>

                {/* Log out */}
                <div className="mt-4 pt-4 border-t border-gray-200">
                  <button
                    onClick={handleLogout}
                    className="w-full flex items-center gap-3 px-4 py-3 rounded-lg text-red-600 hover:bg-red-50 transition-colors"
                  >
                    <span className="text-sm font-medium">Log out</span>
                  </button>
                </div>
              </nav>
            </div>
          </aside>
        </>
      )}
    </header>
  );
}
