# API Integration Guide

This document describes the top-tier API integration layer for the Fresha platform.

## Architecture Overview

### Core Principles

1. **All API functions live in `src/features/<feature>/api/`**
2. **All validation uses Zod schemas**
3. **All requests use the fetcher wrapper**
4. **All tokens are read-only on the server**
5. **All APIs return normalized camelCase data**
6. **UI never calls fetch directly**
7. **React Query hooks for all API calls**
8. **Enforce SSR/CSR/ISR strategies**

## File Structure

```
src/
├── lib/
│   ├── api/
│   │   ├── fetcher.ts          # Enhanced fetcher with token management
│   │   └── auth.ts             # Server-side auth helpers
│   ├── schemas/
│   │   ├── venue.ts            # Venue Zod schemas
│   │   ├── service.ts          # Service Zod schemas
│   │   ├── review.ts          # Review Zod schemas
│   │   ├── booking.ts         # Booking Zod schemas
│   │   └── index.ts           # Central exports
│   └── react-query/
│       └── provider.tsx        # React Query setup
└── features/
    ├── venues/
    │   ├── api/
    │   │   ├── getVenue.ts           # Server + Client functions
    │   │   ├── getVenueServices.ts
    │   │   └── getVenueReviews.ts
    │   └── hooks/
    │       ├── useVenue.ts           # React Query hooks
    │       ├── useVenueServices.ts
    │       └── useVenueReviews.ts
    ├── booking/
    │   ├── api/
    │   │   ├── createBooking.ts
    │   │   └── getAvailableSlots.ts
    │   └── hooks/
    │       ├── useCreateBooking.ts
    │       └── useAvailableSlots.ts
    └── search/
        ├── api/
        │   └── searchVenues.ts
        └── hooks/
            └── useSearchVenues.ts
```

## Fetcher Wrapper

### Server-side Usage

```typescript
import { serverFetcher } from "@/lib/api/fetcher-server";

// In Server Components, Server Actions, Route Handlers
const data = await serverFetcher<Venue>("/venues/slug", {
  requireAuth: false,
  method: "GET",
});
```

**Important:** Only import `fetcher-server` in server-side code. It uses `next/headers` which only works in Server Components.

### Client-side Usage

```typescript
import { clientFetcher } from "@/lib/api/fetcher-client";

// In Client Components, React Query hooks
const data = await clientFetcher<Venue>("/venues/slug", {
  requireAuth: false,
  method: "GET",
});
```

**Note:** `fetcher-client` is marked with `"use client"` and only works in client components.

### Features

- **Automatic token management**: Reads from cookies (server) or localStorage (client)
- **Data normalization**: Converts snake_case to camelCase automatically
- **Error handling**: Throws `ApiError` with status codes
- **Type safety**: Full TypeScript support

## Zod Schemas

All API responses are validated using Zod schemas:

```typescript
import { VenueSchema } from "@/lib/schemas/venue";

const data = await serverFetcher<unknown>("/venues/slug");
const venue = VenueSchema.parse(data); // Validates and types
```

## API Implementation Pattern

Each API function should have both server and client versions:

```typescript
// src/features/venues/api/getVenue.ts
import { serverFetcher, clientFetcher } from "@/lib/api/fetcher";
import { VenueSchema, type Venue } from "@/lib/schemas/venue";

export async function getVenueServer(params: GetVenueParams): Promise<Venue> {
  const data = await serverFetcher<unknown>(`/venues/${params.slug}`, {
    requireAuth: false,
    method: "GET",
  });
  return VenueSchema.parse(data);
}

export async function getVenueClient(params: GetVenueParams): Promise<Venue> {
  const data = await clientFetcher<unknown>(`/venues/${params.slug}`, {
    requireAuth: false,
    method: "GET",
  });
  return VenueSchema.parse(data);
}
```

## React Query Hooks

Each API function should have a matching React Query hook:

```typescript
// src/features/venues/hooks/useVenue.ts
import { useQuery } from "@tanstack/react-query";
import { getVenueClient } from "../api/getVenue";

export function useVenue(params: UseVenueParams, options?) {
  return useQuery({
    queryKey: ["venue", params.slug],
    queryFn: () => getVenueClient(params),
    ...options,
  });
}
```

## Usage Examples

### Server Component (SSR)

```typescript
// app/venues/[slug]/page.tsx
import { getVenueServer } from "@/features/venues/api/getVenue";

export default async function VenuePage({ params }) {
  const venue = await getVenueServer({ slug: params.slug });
  return <div>{venue.name}</div>;
}
```

### Client Component (CSR)

```typescript
// components/VenueCard.tsx
"use client";
import { useVenue } from "@/features/venues/hooks/useVenue";

export function VenueCard({ slug }) {
  const { data: venue, isLoading } = useVenue({ slug });

  if (isLoading) return <div>Loading...</div>;
  return <div>{venue.name}</div>;
}
```

### Mutation Example

```typescript
// components/BookingForm.tsx
"use client";
import { useCreateBooking } from "@/features/booking/hooks/useCreateBooking";

export function BookingForm() {
  const mutation = useCreateBooking({
    onSuccess: (booking) => {
      router.push(`/bookings/${booking.id}`);
    },
  });

  const handleSubmit = (data) => {
    mutation.mutate(data);
  };
}
```

## Authentication

### Server-side Token Access

```typescript
import { getServerToken } from "@/lib/api/auth-server";

// In Server Components, Server Actions, Route Handlers
const token = await getServerToken();
```

**Important:** Only import `auth-server` in server-side code. It uses `next/headers` which only works in Server Components.

### Client-side Token Access

```typescript
import { getClientToken } from "@/lib/api/auth-client";

// In Client Components
const token = getClientToken();
```

**Note:** The fetcher automatically handles server/client token management. You typically don't need to import these directly unless you're building custom auth logic.

## Best Practices

1. **Always validate responses** with Zod schemas
2. **Use server functions in Server Components** for SSR
3. **Use client functions in React Query hooks** for CSR
4. **Never expose sensitive data** to the client
5. **Normalize all API responses** to camelCase
6. **Handle errors gracefully** with proper error boundaries
7. **Use React Query for caching** and state management
8. **Type everything** with TypeScript and Zod

## Migration Guide

To migrate existing APIs:

1. Move API functions to `src/features/<feature>/api/`
2. Add Zod schemas in `src/lib/schemas/`
3. Replace direct fetch calls with `serverFetcher` or `clientFetcher`
4. Validate responses with Zod schemas
5. Create React Query hooks in `src/features/<feature>/hooks/`
6. Update components to use hooks instead of direct API calls
