# Project Rules & Structure Guide

This document defines the complete project structure, conventions, and guidelines for code placement in the Fresha clone project.

## 📁 Project Structure

```
src/
├── app/                    # Next.js App Router - Routes & Layouts
│   ├── layout.tsx          # Root layout (Header, global styles)
│   ├── page.tsx            # Home page (/)
│   ├── (auth)/             # Auth route group (clean URLs)
│   │   ├── login/
│   │   └── signup/
│   ├── (customer)/         # Customer route group
│   │   └── profile/
│   │       ├── layout.tsx  # Profile layout with SideNav
│   │       ├── page.tsx    # /profile
│   │       ├── appointments/
│   │       ├── wallets/
│   │       ├── favorites/
│   │       └── settings/
│   ├── (marketing)/        # Marketing route group
│   │   └── business/
│   └── (venues)/           # Venues route group
│       └── [venueSlug]/
│           ├── page.tsx    # /venues/[slug]
│           └── booking/
│               ├── layout.tsx
│               ├── page.tsx
│               ├── select-option/
│               ├── services/
│               ├── professional/
│               ├── time/
│               ├── questions/
│               └── review/
│
├── components/             # Reusable UI Components (Design System)
│   ├── layout/            # Layout components
│   │   ├── AppLayout.tsx
│   │   ├── Container.tsx  # Max-width container
│   │   ├── PageSection.tsx
│   │   └── TwoColumnLayout.tsx
│   │
│   ├── navigation/        # Navigation components
│   │   ├── Header.tsx    # Main site header
│   │   ├── Footer.tsx    # Main site footer
│   │   ├── SideNav.tsx   # Side navigation
│   │   ├── Breadcrumbs.tsx
│   │   ├── BackButton.tsx
│   │   ├── CloseButton.tsx
│   │   └── NavLink.tsx
│   │
│   ├── ui/               # Base UI primitives (shadcn-style)
│   │   ├── button.tsx
│   │   ├── input.tsx
│   │   ├── textarea.tsx
│   │   ├── select.tsx
│   │   ├── checkbox.tsx
│   │   ├── radio-group.tsx
│   │   ├── switch.tsx
│   │   ├── avatar.tsx
│   │   ├── badge.tsx
│   │   ├── card.tsx
│   │   ├── tabs.tsx
│   │   ├── separator.tsx
│   │   ├── tooltip.tsx
│   │   ├── dialog.tsx
│   │   ├── drawer.tsx
│   │   ├── popover.tsx
│   │   ├── scroll-area.tsx
│   │   ├── skeleton.tsx
│   │   ├── toast.tsx
│   │   └── alert.tsx
│   │
│   ├── data-display/     # Data visualization
│   │   ├── StarRating.tsx
│   │   ├── Stat.tsx
│   │   ├── StatGrid.tsx
│   │   ├── List.tsx
│   │   ├── Table.tsx
│   │   ├── EmptyState.tsx
│   │   ├── KeyValueList.tsx
│   │   └── ImageWithAspectRatio.tsx
│   │
│   ├── forms/            # Form components
│   │   ├── FormField.tsx
│   │   ├── FormGroup.tsx
│   │   ├── SearchField.tsx
│   │   ├── CardNumberField.tsx
│   │   ├── ExpiryField.tsx
│   │   └── CVCField.tsx
│   │
│   ├── feedback/         # User feedback
│   │   ├── Spinner.tsx
│   │   └── InlineMessage.tsx
│   │
│   ├── overlays/         # Overlay components
│   │   └── ConfirmDialog.tsx
│   │
│   ├── media/            # Media components
│   │   ├── PhoneMockup.tsx
│   │   └── DashboardMockup.tsx
│   │
│   ├── carousel/         # Carousel components
│   │   ├── HorizontalCarousel.tsx
│   │   └── CarouselArrow.tsx
│   │
│   └── icons/           # Icon exports
│       └── index.ts
│
├── features/             # Feature-based Components (Domain Logic)
│   ├── home/            # Home page features
│   │   ├── components/
│   │   │   ├── HeroSection.tsx
│   │   │   ├── HeroSearchBar.tsx
│   │   │   ├── HeroSearchSegmentField.tsx
│   │   │   ├── RecommendedSection.tsx
│   │   │   ├── NewToFreshaSection.tsx
│   │   │   ├── TrendingSection.tsx
│   │   │   ├── AppDownloadSection.tsx
│   │   │   ├── ReviewsSection.tsx
│   │   │   ├── TopStatsSection.tsx
│   │   │   ├── BusinessPromoSection.tsx
│   │   │   ├── CityDirectorySection.tsx
│   │   │   └── VenueCard.tsx
│   │   ├── hooks/
│   │   │   └── useHomepageData.ts
│   │   └── api/
│   │       └── getHomeContent.ts
│   │
│   ├── venues/          # Venue-related features
│   │   ├── components/
│   │   │   ├── VenuePageLayout.tsx
│   │   │   ├── VenueHeader.tsx
│   │   │   ├── VenueGallery.tsx
│   │   │   ├── VenueActionsPanel.tsx
│   │   │   ├── VenueTabbedServices.tsx
│   │   │   ├── ServiceList.tsx
│   │   │   ├── ServiceListItem.tsx
│   │   │   ├── TeamSection.tsx
│   │   │   ├── StaffCard.tsx
│   │   │   ├── VenueReviewsSection.tsx
│   │   │   ├── VenueAboutSection.tsx
│   │   │   ├── OpeningHoursTable.tsx
│   │   │   ├── OtherLocationsSection.tsx
│   │   │   ├── NearbyVenuesSection.tsx
│   │   │   ├── CategoryLinksSection.tsx
│   │   │   └── VenueCard.tsx
│   │   ├── api/
│   │   │   ├── getVenue.ts
│   │   │   ├── getVenueServices.ts
│   │   │   └── getVenueReviews.ts
│   │   └── types/
│   │       ├── venue.ts
│   │       └── service.ts
│   │
│   ├── booking/         # Booking flow features
│   │   ├── components/
│   │   │   ├── BookingStepperLayout.tsx
│   │   │   ├── BookingSummaryCard.tsx
│   │   │   ├── ChooseOptionSection.tsx
│   │   │   ├── OptionCard.tsx
│   │   │   ├── ServicesStep.tsx
│   │   │   ├── ServiceSelectableCard.tsx
│   │   │   ├── ServiceCategoryTabs.tsx
│   │   │   ├── ProfessionalStep.tsx
│   │   │   ├── ProfessionalList.tsx
│   │   │   ├── ProfessionalCard.tsx
│   │   │   ├── TimeStep.tsx
│   │   │   ├── DateScroller.tsx
│   │   │   ├── TimeSlotList.tsx
│   │   │   ├── TimeSlotButton.tsx
│   │   │   ├── QuestionStep.tsx
│   │   │   ├── AnswerCard.tsx
│   │   │   ├── ReviewConfirmStep.tsx
│   │   │   ├── PaymentMethodForm.tsx
│   │   │   ├── DiscountCodeField.tsx
│   │   │   ├── PolicyBlock.tsx
│   │   │   ├── BookingNotesField.tsx
│   │   │   └── PriceBreakdownList.tsx
│   │   ├── hooks/
│   │   │   └── useBookingState.ts
│   │   ├── api/
│   │   │   ├── createBooking.ts
│   │   │   ├── getAvailableSlots.ts
│   │   │   └── getProfessionals.ts
│   │   └── types/
│   │       └── booking.ts
│   │
│   ├── auth/            # Authentication features
│   │   ├── components/
│   │   │   ├── AuthLayoutSplit.tsx
│   │   │   ├── AuthHeader.tsx
│   │   │   ├── SocialLoginButton.tsx
│   │   │   ├── OrDivider.tsx
│   │   │   ├── EmailAuthForm.tsx
│   │   │   └── AuthFooterLinks.tsx
│   │   ├── api/
│   │   │   ├── loginWithEmail.ts
│   │   │   └── loginWithProvider.ts
│   │   ├── hooks/
│   │   │   └── useAuth.ts
│   │   └── types/
│   │       └── user.ts
│   │
│   ├── profile/         # User profile features
│   │   ├── components/
│   │   │   ├── ProfileLayout.tsx
│   │   │   ├── ProfileCard.tsx
│   │   │   ├── EditableAvatar.tsx
│   │   │   ├── AddressesCard.tsx
│   │   │   └── AddressTile.tsx
│   │   ├── api/
│   │   │   ├── getCurrentUser.ts
│   │   │   └── updateProfile.ts
│   │   └── hooks/
│   │       └── useProfile.ts
│   │
│   └── search/          # Search & filter features
│       ├── components/
│       │   ├── GlobalHeaderSearchBar.tsx
│       │   └── FiltersDrawer.tsx
│       ├── hooks/
│       │   └── useSearchQueryState.ts
│       └── api/
│           └── searchVenues.ts
│
├── lib/                 # Utility Functions (Non-UI)
│   ├── fetcher.ts       # API fetch wrapper
│   ├── formatCurrency.ts
│   ├── formatDate.ts
│   ├── formatTime.ts
│   ├── classNames.ts
│   ├── analytics.ts
│   ├── auth.ts          # Auth utilities
│   └── validation/
│       ├── bookingValidation.ts
│       └── authValidation.ts
│
├── hooks/               # Shared React Hooks
│   ├── useMediaQuery.ts
│   ├── useToggle.ts
│   ├── useScrollLock.ts
│   └── useDebounce.ts
│
├── config/              # Configuration Files
│   ├── env.ts           # Environment variables
│   ├── routes.ts        # Route constants & builders
│   └── constants.ts     # App-wide constants
│
├── types/               # TypeScript Type Definitions
│   ├── common.ts        # Generic types (ID, Timestamp, Paginated<T>)
│   └── venue.ts         # Venue-related types
│
├── store/               # State Management (Zustand)
│   └── useStore.ts      # Global stores
│
├── styles/              # Global Styles
│   ├── globals.css      # Main global styles
│   ├── tailwind.css     # Tailwind imports
│   ├── typography.css   # Typography styles
│   └── themes.css       # Theme variables
│
├── mocks/               # API Mocks / Storybook Data
│   └── (mock data files)
│
└── tests/               # E2E / Integration Tests
    └── (test files)

public/
├── images/
│   ├── hero/
│   ├── venues/
│   └── mockups/
├── icons/
└── favicons/
```

## 🎯 Code Placement Rules

### 1. Components (`src/components/`)

**Purpose**: Reusable, generic UI components that don't know about business logic.

**Rules**:

- ✅ **DO**: Place generic, reusable components here
- ✅ **DO**: Components should be framework-agnostic (could work in any React app)
- ✅ **DO**: Use TypeScript interfaces for props
- ❌ **DON'T**: Include business logic or domain-specific code
- ❌ **DON'T**: Import from `features/` directory
- ❌ **DON'T**: Hard-code Fresha-specific data

**Examples**:

- `Button`, `Card`, `Input` → `src/components/ui/`
- `Header`, `Footer` → `src/components/navigation/`
- `StarRating`, `Stat` → `src/components/data-display/`

### 2. Features (`src/features/`)

**Purpose**: Domain-aware components that contain business logic specific to Fresha.

**Rules**:

- ✅ **DO**: Place all Fresha-specific components here
- ✅ **DO**: Organize by feature domain (home, venues, booking, auth, etc.)
- ✅ **DO**: Each feature can have:
  - `components/` - Feature-specific components
  - `hooks/` - Feature-specific hooks
  - `api/` - API functions for this feature
  - `types/` - TypeScript types for this feature
- ✅ **DO**: Features can import from `components/` and `lib/`
- ❌ **DON'T**: Import from other features directly
- ❌ **DON'T**: Place generic reusable components here

**Examples**:

- `HeroSection`, `VenueCard` → `src/features/home/components/`
- `VenueHeader`, `ServiceList` → `src/features/venues/components/`
- `BookingStepperLayout` → `src/features/booking/components/`

### 3. Routes (`src/app/`)

**Purpose**: Next.js App Router pages and layouts.

**Rules**:

- ✅ **DO**: Keep route files thin - they should mostly compose feature components
- ✅ **DO**: Use route groups `(auth)`, `(customer)`, `(marketing)`, `(venues)` for organization
- ✅ **DO**: Route groups don't affect URLs (e.g., `(auth)/login` → `/login`)
- ✅ **DO**: Create `layout.tsx` for shared layouts within route groups
- ❌ **DON'T**: Put business logic directly in route files
- ❌ **DON'T**: Create deeply nested route structures unnecessarily

**Examples**:

- Home page → `src/app/page.tsx` (composes `HeroSection`)
- Login → `src/app/(auth)/login/page.tsx`
- Venue detail → `src/app/(venues)/[venueSlug]/page.tsx`
- Booking flow → `src/app/(venues)/[venueSlug]/booking/*/page.tsx`

### 4. Utilities (`src/lib/`)

**Purpose**: Pure utility functions that don't depend on React or UI.

**Rules**:

- ✅ **DO**: Place pure functions here (formatting, validation, etc.)
- ✅ **DO**: Functions should be testable in isolation
- ✅ **DO**: Can be used by both components and features
- ❌ **DON'T**: Include React hooks or components
- ❌ **DON'T**: Include UI-specific code

**Examples**:

- `formatCurrency()`, `formatDate()` → `src/lib/`
- `fetcher()` → `src/lib/fetcher.ts`
- Validation functions → `src/lib/validation/`

### 5. Hooks (`src/hooks/`)

**Purpose**: Shared React hooks used across multiple features.

**Rules**:

- ✅ **DO**: Place reusable hooks here
- ✅ **DO**: Hooks should be generic and reusable
- ❌ **DON'T**: Place feature-specific hooks here (use `features/[feature]/hooks/`)
- ❌ **DON'T**: Include business logic

**Examples**:

- `useToggle`, `useDebounce` → `src/hooks/`
- `useBookingState` → `src/features/booking/hooks/`

### 6. Types (`src/types/`)

**Purpose**: Shared TypeScript type definitions.

**Rules**:

- ✅ **DO**: Place generic, shared types here
- ✅ **DO**: Feature-specific types go in `features/[feature]/types/`
- ❌ **DON'T**: Duplicate types across features

**Examples**:

- `ID`, `Timestamp`, `Paginated<T>` → `src/types/common.ts`
- `Venue` → `src/types/venue.ts` (if shared) or `src/features/venues/types/venue.ts` (if feature-specific)

### 7. Config (`src/config/`)

**Purpose**: Configuration constants and route builders.

**Rules**:

- ✅ **DO**: Place app-wide constants here
- ✅ **DO**: Use `routes.ts` for route constants and URL builders
- ✅ **DO**: Use `constants.ts` for app-wide constants (currencies, limits, etc.)
- ✅ **DO**: Use `env.ts` for environment variable access

**Examples**:

- Route builders → `src/config/routes.ts`
- Currency symbols → `src/config/constants.ts`

### 8. Store (`src/store/`)

**Purpose**: Global state management (Zustand stores).

**Rules**:

- ✅ **DO**: Place global stores here
- ✅ **DO**: Keep stores focused and separate by domain
- ❌ **DON'T**: Mix unrelated state in one store
- ❌ **DON'T**: Use for feature-specific state (use feature hooks instead)

**Examples**:

- Search state → `src/store/useStore.ts` (useSearchStore)
- Carousel state → `src/store/useStore.ts` (useCarouselStore)

## 📝 Naming Conventions

### Files & Directories

- **Components**: PascalCase (`HeroSection.tsx`, `VenueCard.tsx`)
- **Utilities**: camelCase (`formatCurrency.ts`, `useToggle.ts`)
- **Types**: camelCase (`venue.ts`, `booking.ts`)
- **Routes**: lowercase (`page.tsx`, `layout.tsx`)
- **Directories**: lowercase with hyphens if needed (`data-display/`, `booking-flow/`)

### Components

- **Export**: Named exports for components (`export function HeroSection()`)
- **Default exports**: Only for route pages (`export default function HomePage()`)

### Imports

- **Always use `@/` alias**: `import { Button } from "@/components/ui/button"`
- **Group imports**: External → Internal → Relative
- **Absolute paths**: Always use `@/` for `src/` imports

## 🔄 Import Rules

### Allowed Import Patterns

```typescript
// ✅ Components can import from:
import { Button } from "@/components/ui/button";
import { formatCurrency } from "@/lib/formatCurrency";
import { useToggle } from "@/hooks/useToggle";
import { routes } from "@/config/routes";

// ✅ Features can import from:
import { Button } from "@/components/ui/button"; // Generic components
import { formatCurrency } from "@/lib/formatCurrency"; // Utilities
import { useToggle } from "@/hooks/useToggle"; // Shared hooks
import { Venue } from "@/types/venue"; // Shared types

// ✅ Routes can import from:
import { HeroSection } from "@/features/home/components/HeroSection";
import { Container } from "@/components/layout/Container";
```

### Forbidden Import Patterns

```typescript
// ❌ DON'T: Features importing from other features
import { VenueCard } from "@/features/home/components/VenueCard"; // In venues feature

// ❌ DON'T: Components importing from features
import { HeroSection } from "@/features/home/components/HeroSection"; // In Button component

// ❌ DON'T: Relative imports across major boundaries
import { Button } from "../../components/ui/button"; // Use @/ instead
```

## 🎨 Component Structure

### Generic Component Template

```typescript
// src/components/ui/button.tsx
import { ButtonHTMLAttributes, ReactNode } from "react";
import { cn } from "@/lib/classNames";

interface ButtonProps extends ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: "primary" | "secondary";
  children: ReactNode;
}

export function Button({
  variant = "primary",
  className,
  children,
  ...props
}: ButtonProps) {
  return (
    <button
      className={cn("base-classes", variantClasses[variant], className)}
      {...props}
    >
      {children}
    </button>
  );
}
```

### Feature Component Template

```typescript
// src/features/home/components/HeroSection.tsx
"use client";

import { Container } from "@/components/layout/Container";
import { Button } from "@/components/ui/button";
import { useHomepageData } from "../hooks/useHomepageData";

export function HeroSection() {
  const { data } = useHomepageData();

  return (
    <section>
      <Container>{/* Component content */}</Container>
    </section>
  );
}
```

## 🚀 Route Structure

### Route File Template

```typescript
// src/app/(venues)/[venueSlug]/page.tsx
import { VenuePageLayout } from "@/features/venues/components/VenuePageLayout";
import { Container } from "@/components/layout/Container";

interface VenuePageProps {
  params: {
    venueSlug: string;
  };
}

export default function VenuePage({ params }: VenuePageProps) {
  return (
    <main>
      <VenuePageLayout venueSlug={params.venueSlug} />
    </main>
  );
}
```

## 📦 State Management

### When to Use What

- **Zustand Store** (`src/store/`): Global state shared across features
- **Feature Hooks** (`features/[feature]/hooks/`): Feature-specific state
- **React State**: Component-local state
- **URL State**: For filters, search params (use Next.js `useSearchParams`)

## 🧪 Testing Structure

- **Unit Tests**: Co-locate with components or in `__tests__/` folders
- **Integration Tests**: `src/tests/integration/`
- **E2E Tests**: `src/tests/e2e/`

## 📚 File Organization Checklist

Before creating a new file, ask:

1. **Is it reusable across features?** → `src/components/`
2. **Is it specific to one feature?** → `src/features/[feature]/`
3. **Is it a route/page?** → `src/app/`
4. **Is it a pure utility?** → `src/lib/`
5. **Is it a shared hook?** → `src/hooks/`
6. **Is it a type definition?** → `src/types/` or `features/[feature]/types/`
7. **Is it configuration?** → `src/config/`
8. **Is it global state?** → `src/store/`

## 🔍 Quick Reference

| What                   | Where                                   |
| ---------------------- | --------------------------------------- |
| Reusable UI component  | `src/components/ui/`                    |
| Navigation component   | `src/components/navigation/`            |
| Home page section      | `src/features/home/components/`         |
| Venue detail component | `src/features/venues/components/`       |
| Booking step component | `src/features/booking/components/`      |
| Format currency        | `src/lib/formatCurrency.ts`             |
| Shared hook            | `src/hooks/`                            |
| Route constant         | `src/config/routes.ts`                  |
| Home page route        | `src/app/page.tsx`                      |
| Login route            | `src/app/(auth)/login/page.tsx`         |
| Venue route            | `src/app/(venues)/[venueSlug]/page.tsx` |

## ⚠️ Important Notes

1. **Never import from old `components/` or `app/` directories** - Everything is in `src/`
2. **Always use `@/` alias** for imports from `src/`
3. **Route groups `()` don't affect URLs** - Use for organization only
4. **Keep routes thin** - They compose feature components
5. **Components are generic** - Features contain business logic
6. **One feature per directory** - Don't mix features

---

**Last Updated**: 2025-01-XX
**Project**: Fresha Clone
**Framework**: Next.js 14 (App Router) + TypeScript + Tailwind CSS



